/* **********************************************************
 * Copyright 2011-2012, 2017 VMware, Inc. All rights reserved. -- VMware Confidential
 * *********************************************************
 */

package com.vmware.vapi.core;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import com.vmware.vapi.data.DataDefinition;
import com.vmware.vapi.data.ErrorDefinition;
import com.vmware.vapi.internal.protocol.server.rpc.http.util.StringUtil;

/**
 * The <code>MethodDefinition</code> class contains detailed information
 * about a vAPI method. This should contain all the information
 * required to address a method in the vAPI runtime.
 */
public class MethodDefinition {
    private MethodIdentifier id;
    private DataDefinition input;
    private DataDefinition output;
    private Set<ErrorDefinition> errorDefs;
    private Map<String, ErrorDefinition> errorDefsMap;

    /**
     * Levels of task support by the method
     */
    public enum TaskSupport {
        NONE,
        TASK_ENABLED,
        TASK_ONLY,
    }

    private TaskSupport taskSupport;

    /**
     * Constructor.
     *
     * @param id           method identifier
     * @param input        data definition of the method input
     * @param output       data definition of the method output
     * @param errors       set of <code>ErrorDefinition</code>s for errors
     *                     that can be reported by the method; might be
     *                     <code>null</code>
     */
    public MethodDefinition(MethodIdentifier id,
                         DataDefinition input,
                         DataDefinition output,
                         Set<ErrorDefinition> errors) {
        this(id, input, output, errors, TaskSupport.NONE);
    }

    /**
     * Constructor.
     *
     * @param id           method identifier
     * @param input        data definition of the method input
     * @param output       data definition of the method output
     * @param errors       set of <code>ErrorDefinition</code>s for errors
     *                     that can be reported by the method; might be
     *                     <code>null</code>
     * @param taskSupport  Task support level of the method
     */
    public MethodDefinition(MethodIdentifier id,
                            DataDefinition input,
                            DataDefinition output,
                            Set<ErrorDefinition> errors,
                            TaskSupport taskSupport) {
        if (id == null) {
            throw new IllegalArgumentException(
                    "Method identifier is required.");
        }
        if (input == null) {
            throw new IllegalArgumentException(
                    "Data definition of the method input is required.");
        }
        if (output == null) {
            throw new IllegalArgumentException(
                    "Data definition of the method output is required.");
        }

        if (errors == null) {
            this.errorDefs = Collections.<ErrorDefinition>emptySet();
            this.errorDefsMap = Collections.<String, ErrorDefinition>emptyMap();
        } else {
            this.errorDefs = Collections.unmodifiableSet(
                    new HashSet<ErrorDefinition>(errors));
            this.errorDefsMap = Collections.unmodifiableMap(
                    createErrorMap(errors));
        }

        this.id = id;
        this.input = input;
        this.output = output;
        this.taskSupport = taskSupport;
    }

    /**
     * Returns the identifier of this method.
     *
     * @return  method identifier of this method
     */
    public MethodIdentifier getIdentifier() {
        return id;
    }

    /**
     * Returns the input definition of this method.
     *
     * @return  data definition of the input of this method
     */
    public DataDefinition getInputDefinition() {
        return input;
    }

    /**
     * Returns the output definition of this method.
     *
     * @return  data definition of the output of this method
     */
    public DataDefinition getOutputDefinition() {
        return output;
    }

    /**
     * Returns a set of error definitions describing the errors
     * which this method can report.
     *
     * @return set of <code>ErrorDefinition</code>s for the method
     */
    public Set<ErrorDefinition> getErrorDefinitions() {
        return Collections.unmodifiableSet(errorDefs);
    }

    /**
     * Returns the error definition with the specified name reported by this
     * method or <code>null</code> if this method doesn't report an error with
     * the specified name.
     *
     * @param errorName  name of the error definition to return
     * @return error definition with the specified name reported by this method
     *         or <code>null</code> if this method doesn't report an error with
     *         the specified name.
     * @throws NullPointerException if the error name is <code>null</code>
     */
    public ErrorDefinition getErrorDefinition(String errorName) {
        if (errorName == null) {
            throw new NullPointerException();
        }
        return errorDefsMap.get(errorName);
    }

    /**
     * Returns task support level of the method.
     *
     * @return TaskSupport enum to indicate the task support level of the method
     */
    public TaskSupport getTaskSupport() {
        return taskSupport;
    }

    /**
     * Creates a map from the specified set of error definitions, where the
     * keys are the error names and the values are the error definitions.
     *
     * @param errorDefs  set of error definitions
     * @return mapping of error name to error definition
     */
    private static Map<String, ErrorDefinition> createErrorMap(
            Set<ErrorDefinition> errorDefs) {
        Map<String, ErrorDefinition> errorMap =
                new HashMap<String, ErrorDefinition>();
        for (ErrorDefinition def : errorDefs) {
            errorMap.put(def.getName(), def);
        }
        return errorMap;
    }

    @Override
    public String toString() {
        return "Method id: " + id.toString() + "\n" +
                " Input: " + input.toString() + "\n" +
                " Output: " + output.toString() + "\n" +
                " Errors: " + errorDefs.toString() + "\n" +
                " taskSupport: " + taskSupport.name();


    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (!(obj instanceof MethodDefinition)) {
            return false;
        }
        MethodDefinition other = (MethodDefinition) obj;
        return getIdentifier().equals(other.getIdentifier()) &&
                getInputDefinition().equals(other.getInputDefinition()) &&
                getOutputDefinition().equals(other.getOutputDefinition()) &&
                getErrorDefinitions().equals(other.getErrorDefinitions()) &&
                getTaskSupport() == other.getTaskSupport();
    }

    @Override
    public int hashCode() {
        int hash = 1;
        hash = hash * 31 + getIdentifier().hashCode();
        hash = hash * 31 + getInputDefinition().hashCode();
        hash = hash * 31 + getOutputDefinition().hashCode();
        hash = hash * 31 + getErrorDefinitions().hashCode();
        hash = hash * 31 + getTaskSupport().hashCode();

        return hash;
    }

}