/* **********************************************************
 * Copyright 2011-2012 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * DataDefinition.java --
 *
 *      Abstract base class for all types in the vAPI run-time type system.
 */

package com.vmware.vapi.data;

import java.util.LinkedList;
import java.util.List;

import com.vmware.vapi.*;

/**
 * The <code>DataDefinition</code> class is an abstract base class for all
 * types in the vAPI run-time type system.
 */
public abstract class DataDefinition {

    /**
      * Constructor for (typically implicit) use by derived class constructors.
      */
     protected DataDefinition() {}

    /**
     * Returns the {@link DataType} for this type.
     *
     * @return  the {@link DataType} for this type
     */
    public abstract DataType getType();

    /**
     * Applies a visitor to this data-definition.
     *
     * @param visitor the visitor operating on this data-definition
     */
    public abstract void accept(DefinitionVisitor visitor);

    /**
     * Validates that the specified {@link DataValue} is an instance of this
     * data-definition.
     *
     * @param value  the data value to validate
     * @return       true if the value is an instance of this data definition,
     *               false otherwise
     */
    public boolean validInstanceOf(DataValue value) {
        List<Message> errors = validate(value);
        return errors.isEmpty();
    }

    /**
     * Validates that the specified {@link DataValue} is an instance of this
     * data definition.
     *
     * <p>Validates that supplied <code>value</code> is not <code>null</code>
     * and it's type matches the type of this definition.
     *
     * @param value  the <code>DataValue</code> to validate
     * @return       a list of messages describing validation problems; empty
     *               list indicates that validation was successful
     */
    public List<Message> validate(DataValue value) {
        List<Message> result = new LinkedList<Message>();
        if (value == null) {
            result.add(MessageFactory.getMessage("vapi.data.validate.mismatch",
                                                 getType().toString(),
                                                 "null"));
        } else if (value.getType() != getType()) {
            result.add(MessageFactory.getMessage("vapi.data.validate.mismatch",
                                                 getType().toString(),
                                                 value.getType().toString()));
        }
        return result;
    }

    /**
     * Check the value to see if the data-definition can fill in any missing
     * data before validation
     *
     * @param value  the value to check
     */
    public void completeValue(DataValue value) {
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String toString() {
        return getType().toString();
    }
}
