/* **********************************************************
 * Copyright (c) 2011-2014, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.data;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import com.vmware.vapi.CoreException;
import com.vmware.vapi.internal.util.Validate;

/**
 * <i>Thread-safety:</i> Instances of this class are not thread-safe.
 */
public class StructValue implements DataValue {

    private static final long serialVersionUID = 1L;
    private final HashMap<String, DataValue> fields;
    private final String name;

    /**
     * @param name the canonical name of the structure type. cannot be
     *        <code>null</code>
     */
    public StructValue(String name) {
        Validate.notNull(name);
        this.name = name;
        fields = new HashMap<>();
    }

    @Override
    public DataType getType() {
        return DataType.STRUCTURE;
    }

    public String getName() {
        return name;
    }

    public Set<String> getFieldNames() {
        return fields.keySet();
    }

    public boolean hasField(String field) {
        return fields.containsKey(field);
    }

    /**
     * Gets the value for a field with given name.
     *
     * @param field  name of the field
     * @return  <code>DataValue</code> representing the value of the field
     *
     * @throws CoreException  if field with the specified name is not present
     */
    public DataValue getField(String field) {
        if (fields.containsKey(field)) {
            return fields.get(field);
        }

        throw new CoreException("vapi.data.structure.getfield.unknown",
                                field);
    }

    public Map<String, DataValue> getFields() {
        return Collections.unmodifiableMap(fields);
    }

    public long getInteger(String field) {
        DataValue value = getField(field);

        if (value instanceof IntegerValue) {
            IntegerValue tmp = (IntegerValue) value;
            return tmp.getValue();
        } else {
            throw new CoreException("vapi.data.structure.getfield.mismatch",
                                    field,
                                    "Integer",
                                    value.getType().toString());
        }
    }

    public double getDouble(String field) {
        DataValue value = getField(field);

        if (value instanceof DoubleValue) {
            DoubleValue tmp = (DoubleValue) value;
            return tmp.getValue();
        } else {
            throw new CoreException("vapi.data.structure.getfield.mismatch",
                                    field,
                                    "Double",
                                    value.getType().toString());
        }
    }

    public boolean getBoolean(String field) {
        DataValue value = getField(field);

        if (value instanceof BooleanValue) {
            BooleanValue tmp = (BooleanValue) value;
            return tmp.getValue();
        } else {
            throw new CoreException("vapi.data.structure.getfield.mismatch",
                                    field,
                                    "Boolean",
                                    value.getType().toString());
        }
    }

    /**
     *
     * @param fieldName name of the field to fetch
     * @return String representing the value of the field
     *
     * @throws CoreException if field with the specified name is not present
     *                       or its type is not <code>StringValue</code>
     */
    public String getString(String fieldName) {
        DataValue value = getField(fieldName);

        if (value instanceof StringValue) {
            return ((StringValue) value).getValue();
        }

        throw new CoreException("vapi.data.structure.getfield.mismatch",
                                fieldName,
                                "String",
                                value.getType().toString());
    }

    public char[] getSecret(String field) {
        DataValue value = getField(field);

        if (value instanceof SecretValue) {
            return ((SecretValue) value).getValue();
        }

        throw new CoreException("vapi.data.structure.getfield.mismatch",
                                field,
                                "Secret",
                                value.getType().toString());
    }

    public OptionalValue getOptional(String field) {
        DataValue value = getField(field);

        if (value instanceof OptionalValue) {
            return (OptionalValue) value;
        }

        throw new CoreException("vapi.data.structure.getfield.mismatch",
                                field,
                                "Optional",
                                value.getType().toString());
    }

    /**
     * Gets the list value for a field with given name.
     *
     * @param field  name of the field
     * @return  <code>ListValue</code> representing the value of the field
     *
     * @throws CoreException  if field with the specified name is not present
     *                        or its type is not list
     */
    public ListValue getList(String field) {
        DataValue value = getField(field);
        if (value instanceof ListValue) {
            return (ListValue) value;
        }

        throw new CoreException("vapi.data.structure.getfield.mismatch",
                                field,
                                "List",
                                value.getType().toString());
    }

    public StructValue getStruct(String field) {
        DataValue value = getField(field);
        if (value instanceof StructValue) {
            return (StructValue) value;
        }

        throw new CoreException("vapi.data.structure.getfield.mismatch",
                                field,
                                "Struct",
                                value.getType().toString());
    }

    public void setField(String field, DataValue value) {
        if (field == null || value == null) {
            throw new CoreException("vapi.data.structure.setfield.null",
                                    field);
        }

        fields.put(field, value);
    }

    public void setField(String field, long value) {
        setField(field, new IntegerValue(value));
    }

    public void setField(String field, boolean value) {
        setField(field, BooleanValue.getInstance(value));
    }

    public void setField(String field, double value) {
        setField(field, new DoubleValue(value));
    }

    public void setField(String field, String value) {
        setField(field, new StringValue(value));
    }

    public void setField(String field, char[] value) {
        setField(field, new SecretValue(value));
    }

    public void setOptionalField(String field, Long value) {
        setField(field, new OptionalValue(value));
    }

    public void
            setOptionalField(String field, Boolean value) {
        setField(field, new OptionalValue(value));
    }

    public void
            setOptionalField(String field, Double value) {
        setField(field, new OptionalValue(value));
    }

    public void
            setOptionalField(String field, String value) {
        setField(field, new OptionalValue(value));
    }

    public void setOptionalField(String field, long value) {
        setField(field, new OptionalValue(value));
    }

    public void setOptionalField(String field, boolean value) {
        setField(field, new OptionalValue(value));
    }

    public void setOptionalField(String field, double value) {
        setField(field, new OptionalValue(value));
    }

    @Override
    public boolean equals(Object other) {
        // this also handles other == null as well
        if (!StructValue.class.isInstance(other)) {
            return false;
        }

        StructValue strOther = (StructValue) other;
        if (!strOther.name.equals(name)) {
            return false;
        }

        return fields.equals(strOther.fields);
    }

    @Override
    public int hashCode() {
        int hash = 17 + name.hashCode();
        hash = hash * 31 + fields.hashCode();
        return hash;
    }

    @Override
    public String toString() {
        return name + " => " + fields.toString();
    }

    @Override
    public void accept(ValueVisitor visitor) {
        visitor.visit(this);
    }

    @Override
    public StructValue copy() {
        StructValue result = new StructValue(this.name);
        for (String field : this.fields.keySet()) {
            result.setField(field, getField(field).copy());
        }
        return result;
    }
}
