/* **********************************************************
 * Copyright 2013-2014, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.diagnostics;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.slf4j.MDC;

import com.vmware.vapi.internal.util.Validate;

/**
 * SLF4J {@link MDC} based log configurator.
 *
 * <p>Configures log diagnostic context as <code>String</code> key and values
 * pairs. Context is supposed to appear in logging traces by the means
 * supported by Log4j. Context is maintained on a per thread basis.
 *
 * <p>Thread switching (like running task in <code>ThreadPoolExecutor</code>
 * or newly created <code>Thread</code> require explicit action in order to
 * transfer the context to the new thread and later clean it up.
 *
 * @see MDC
 */
public final class Slf4jMDCLogConfigurator implements LogDiagnosticsConfigurator {

    @Override
    public void configureContext(Map<String, String> diagCtx) {
        if (diagCtx == null) {
            return;
        }

        for (String key : diagCtx.keySet()) {
            MDC.put(key, diagCtx.get(key));
        }
    }

    @Override
    public void cleanUpContext() {
        MDC.clear();
    }

    @Override
    public void cleanUpContext(Collection<String> keys) {
        Validate.notNull(keys);

        for (String k : keys) {
            MDC.remove(k);
        }
    }

    @Override
    public Map<String, String> getContext() {
        @SuppressWarnings("rawtypes")
        Map ctx = MDC.getCopyOfContextMap();

        if (ctx == null) {
            return Collections.emptyMap();
        }

        @SuppressWarnings("unchecked")
        Map<String, String> result = new HashMap<String, String>(ctx);

        return result;
    }
}
