/* **********************************************************
 * Copyright 2016, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.bindings;

import java.util.HashMap;
import java.util.Map;

import com.vmware.vapi.internal.util.Validate;

/**
 * Description of API operation in the generated Java bindings.
 *
 * <p>Contains information about operation invocation over <b>REST</b>.
 * That is the {@code @RequestMapping} and related annotations in the
 * <tt>VMODL2</tt> API definition.
 */
public class OperationDef {
    private final String operationId;
    private final String urlTemplate;
    private final String httpMethod;
    private final String acceptHeader;
    private final String contentTypeHeader;

    private String requestBodyField;
    private Map<String, String> pathVar2FieldName;
    private Map<String, String> requestParam2FieldName;
    private Map<String, String> header2FieldName;
    // TODO: rest native: @RequestMapping fixed params & headers

    /**
     * Constructor.
     *
     * @param operationId simple (non-qualified by service) operation identifier
     * @param urlTemplate URL template with placeholders for path variables
     * @param httpMethod HTTP method for REST invocation
     * @param acceptHeader value for the <tt>Accept</tt> HTTP header; may be {@code null}
     * @param contentTypeHeader value for the <tt>Content-Type</tt> HTTP header for
     *                          the REST request; may be {@code null}
     */
    public OperationDef(String operationId,
                        String urlTemplate,
                        String httpMethod,
                        String acceptHeader,
                        String contentTypeHeader) {
        Validate.notNull(operationId);
        Validate.notNull(urlTemplate);
        Validate.notNull(httpMethod);

        this.operationId = operationId;
        this.urlTemplate = urlTemplate;
        this.httpMethod = httpMethod;
        this.acceptHeader = acceptHeader;
        this.contentTypeHeader = contentTypeHeader;
    }

    /**
     * Register operation parameter as the HTTP request body one.
     *
     * @param name the wire name of the field; must not be {@code null}
     */
    public void registerRequestBody(String name) {
        Validate.notNull(name);
        requestBodyField = name;
    }

    // TODO: rest native: optimization: Map is not needed for @PathVariable
    //       we should replace the placeholders in the urlTemplate with the
    //       wire names of the referred fields
    /**
     * Register mapping for path variable.
     *
     * @param pathVarName name of the path variable used in the URL template;
     *                    must not be {@code null}
     * @param name the wire name of the parameter for the path variable value;
     *             must not be {@code null}
     */
    public void registerPathVariable(String pathVarName, String name) {
        Validate.notNull(pathVarName);
        Validate.notNull(name);
        if (pathVar2FieldName == null) {
            pathVar2FieldName = newSmallHashMap();
        }
        pathVar2FieldName.put(pathVarName, name);
    }

    /**
     * Register mapping for request (query string) parameter.
     *
     * @param reqParamName name to be used as key in the query string
     *                     parameter; must not be {@code null}
     * @param name the wire name of the parameter for the value; must
     *             not be {@code null}
     */
    public void registerRequestParam(String reqParamName, String name) {
        Validate.notNull(reqParamName);
        Validate.notNull(name);
        if (requestParam2FieldName == null) {
            requestParam2FieldName = newSmallHashMap();
        }
        requestParam2FieldName.put(reqParamName, name);
    }

    /**
     * Register mapping for HTTP header.
     *
     * @param headerName name of the HTTP header; must not be {@code null}
     * @param name the wire name of the parameter for the header value; must
     *             not be {@code null}
     */
    public void registerRequestHeader(String headerName, String name) {
        Validate.notNull(headerName);
        Validate.notNull(name);
        if (header2FieldName == null) {
            header2FieldName = newSmallHashMap();
        }
        header2FieldName.put(headerName, name);
    }

    public String getOperationId() {
        return operationId;
    }

    public String getUrlTemplate() {
        return urlTemplate;
    }

    public String getHttpMethod() {
        return httpMethod;
    }

    public String getAcceptHeaderValue() {
        return acceptHeader;
    }

    public String getContentTypeHeaderValue() {
        return contentTypeHeader;
    }

    public String getRequestBodyFieldName() {
        return requestBodyField;
    }

    public Map<String, String> getPathVariableInfo() {
        return pathVar2FieldName;
    }

    public Map<String, String> getRequestParamInfo() {
        return requestParam2FieldName;
    }

    public Map<String, String> getHeadersInfo() {
        return header2FieldName;
    }

    private Map<String, String> newSmallHashMap() {
        // we expect really small number of elements in these maps
        // but we need 4 * CAPACITY + 32 * SIZE memory for each instance
        return new HashMap<>(8, 1.0f);
    }
}