/* **********************************************************
 * Copyright (c) 2012-2014, 2017 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.bindings.convert;

import com.vmware.vapi.bindings.type.AnyErrorType;
import com.vmware.vapi.bindings.type.DynamicStructType;
import com.vmware.vapi.bindings.type.EnumType;
import com.vmware.vapi.bindings.type.ErrorType;
import com.vmware.vapi.bindings.type.ListType;
import com.vmware.vapi.bindings.type.MapType;
import com.vmware.vapi.bindings.type.OptionalType;
import com.vmware.vapi.bindings.type.SetType;
import com.vmware.vapi.bindings.type.StructType;
import com.vmware.vapi.data.BlobValue;
import com.vmware.vapi.data.BooleanValue;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.DoubleValue;
import com.vmware.vapi.data.ErrorValue;
import com.vmware.vapi.data.IntegerValue;
import com.vmware.vapi.data.ListValue;
import com.vmware.vapi.data.OptionalValue;
import com.vmware.vapi.data.SecretValue;
import com.vmware.vapi.data.StringValue;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.data.VoidValue;

/**
 * Abstraction of a full suite of converters which cover all IDL types.
 * Each converter handles the conversion of a particular type.
 */
public interface ConverterFactory {

    /**
     * Returns a converter for integer type.
     *
     * @return integer converter
     */
    PrimitiveConverter<IntegerValue> getIntegerConverter();

    /**
     * Returns a converter for double-precision floating point type.
     *
     * @return double converter
     */
    PrimitiveConverter<DoubleValue> getDoubleConverter();

    /**
     * Returns a converter for boolean type.
     *
     * @return boolean converter
     */
    PrimitiveConverter<BooleanValue> getBooleanConverter();

    /**
     * Returns a converter for string type.
     *
     * @return string converter
     */
    PrimitiveConverter<StringValue> getStringConverter();

    /**
     * Returns a converter for binary type.
     *
     * @return binary converter
     */
    PrimitiveConverter<BlobValue> getBinaryConverter();

    /**
     * Returns a converter for secret type.
     *
     * @return secret converter
     */
    PrimitiveConverter<SecretValue> getSecretConverter();

    /**
     * Returns a converter for date-time type.
     *
     * @return date-time converter
     */
    PrimitiveConverter<StringValue> getDateTimeConverter();

    /**
     * Returns a converter for URI type.
     *
     * @return URI converter
     */
    PrimitiveConverter<StringValue> getUriConverter();

    /**
     * Returns a converter for void type.
     *
     * @return void converter
     */
    PrimitiveConverter<VoidValue> getVoidConverter();

    /**
     * Returns a converter for opaque type.
     *
     * @return opaque converter
     */
    PrimitiveConverter<DataValue> getOpaqueConverter();

    /**
     * Returns a converter for dynamic data (<tt>Structure</tt>) type.
     *
     * @return dynamic structure converter
     */
    UniTypeConverter<StructValue, DynamicStructType> getDynamicStructureConverter();

    /**
     * Returns a converter for list type.
     *
     * @return list converter
     */
    UniTypeConverter<ListValue, ListType> getListConverter();

    /**
     * Returns a converter for structure type.
     *
     * @return structure converter
     */
    UniTypeConverter<StructValue, StructType> getStructConverter();

    /**
     * Returns a converter for optional type.
     *
     * @return optional converter
     */
    UniTypeConverter<OptionalValue, OptionalType> getOptionalConverter();

    /**
     * Returns a converter for enumeration type.
     *
     * @return enumeration converter
     */
    UniTypeConverter<StringValue, EnumType> getEnumConverter();

    /**
     * Returns a converter for error type.
     *
     * @return error converter
     */
    UniTypeConverter<ErrorValue, ErrorType> getErrorConverter();

    /**
     * Returns a converter for ID type.
     *
     * @return ID converter
     */
    PrimitiveConverter<StringValue> getIdConverter();

    /**
     * Returns a converter for set type.
     *
     * @return set converter
     */
    UniTypeConverter<ListValue, SetType> getSetConverter();

    /**
     * Returns a converter for map type.
     *
     * @return map converter
     */
    UniTypeConverter<DataValue, MapType> getMapConverter();

    /**
     * Returns a converter for <tt>Exception</tt> IDL type (a.k.a. AnyError).
     *
     * @return AnyError converter
     */
    UniTypeConverter<ErrorValue, AnyErrorType> getAnyErrorConverter();
}
