/* **********************************************************
 * Copyright 2014, 2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.core.abort;

import com.vmware.vapi.core.ApiProvider;
import com.vmware.vapi.core.AsyncHandle;
import com.vmware.vapi.data.DataValue;

/**
 * <p>Handle that can be used to abort execution of a given event. Supports list
 * of {@link AbortListener abort listeners} that will be notified
 * when the event is being aborted. The purpose of this handle is to release all
 * client resources associated with a request and this way effectively to abort
 * the request.</p>
 * <p>The server side will not get notified, the only symptom that
 * something has happened would be the closed connection (in case of HTTP
 * connection). If server side notification is required - the customer can
 * provide such by adding a custom {@link AbortListener} that would handle
 * this.</p>
 * <p> The use pattern for this is:<ol>
 * <li> Create an abort handle.
 * <li> Supply the abort handle to the {@link AbortableAsyncHandle}
 * <li> Use the abortable async handle as last argument of the
 * {@link ApiProvider#invoke(String, String, DataValue, ExecutionContext,
 * AsyncHandle)}.
 * <li> Use the {@link AbortHandle#abort()} to abort the request at the desired
 * moment.
 * </ol></p>
 */
public interface AbortHandle {
    /**
     * Add {@link AbortListener} for this handle. If the handle is aborted
     * all listeners will be notified for that.
     *
     * @param listener the abort listener; must not be null.
     */
    void addAbortListener(AbortListener listener);

    /**
     * Abort the event associated with this handle. This will set the state of
     * the handle to aborted and will notify all registered
     * {@link AbortListener abort listeners}. Once a request is completed (i.e.
     * {@link AsyncHandle#setResult(Object)} or
     * {@link AsyncHandle#setError(RuntimeException)} was invoked) the method
     * call would have no effect.
     * <p>
     * If the request is still active (i.e. not completed and not aborted) the
     * invoke to this method will result in completing the request with an
     * {@link RequestAbortedException} reported to the request
     * {@link AsyncHandle}.
     * <p>
     * This method never throws an {@link Exception}.
     */
    void abort();

    /**
     * Check the state of the handle.
     *
     * @return true iff the handle is aborted.
     */
    boolean isAborted();
}