/* **********************************************************
 * Copyright (c) 2012-2013, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.diagnostics;

import java.lang.management.ManagementFactory;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicLong;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.AttributeNotFoundException;
import javax.management.DynamicMBean;
import javax.management.InvalidAttributeValueException;
import javax.management.MBeanAttributeInfo;
import javax.management.MBeanException;
import javax.management.MBeanInfo;
import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.ReflectionException;

import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.internal.util.time.Chronometer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class DiagnosticsManager implements DynamicMBean {
    private static final Logger logger =
            LoggerFactory.getLogger(DiagnosticsManager.class);
    private static final String ROOT_COMPONENT ="com.vmware.vapi";
    private static final String OBJECT_FORMAT =
            ROOT_COMPONENT + ":" + "type=%s,id=%s";
    private final ConcurrentMap<String, AtomicLong> counters;
    private final List<AtomicLong> stopwatchSamples;
    private final DiagnosticsKey key;
    private int samples=1;

    /**
     * <p>
     * Creates a diagnostic JMX bean and exports it in the default JMX bean
     * server of the platform. (see
     * {@link ManagementFactory#getPlatformMBeanServer()}).
     * </p>
     * <p>
     * Every invocation of this method would register a new distinct JMX bean.
     * The identity of this new JMX bean is determined by the identity of the
     * Java object that backs the bean (see
     * {@link System#identityHashCode(Object)}). As a result there could be
     * multiple beans registered for the same diagnostic key. This approach (no
     * sharing of beans) solves sharing problems caused by separate class
     * loaders (e.g. deploy the vAPI runtime jar simultaneously in multiple .war
     * applications) or having multiple vAPI objects of the same type (e.g.
     * multiple instances of vAPI servlet or LocalProvider).
     * </p>
     *
     * @param diagnosticsKey key which identifies the type of the diagnostic
     *                        bean; must not be <code>null</code>
     * @return JMX bean
     */
    public static DiagnosticsManager createAndExport(
            DiagnosticsKey diagnosticsKey) {
        Validate.notNull(diagnosticsKey);
        DiagnosticsManager bean = new DiagnosticsManager(diagnosticsKey);
        MBeanServer server = ManagementFactory.getPlatformMBeanServer();
        ObjectName name;
        try {
            name = createObjectName(bean);
            server.registerMBean(bean, name);
        } catch (Exception ex) {
            // JMX is not vital - complain but do not throw.
            logger.warn(String.format(
                    "Could not export diagnostics JMX bean '%s'",
                    diagnosticsKey), ex);
        }
        return bean;
    }

    private DiagnosticsManager(DiagnosticsKey diagnosticsKey) {
        key = diagnosticsKey;
        counters = new ConcurrentHashMap<String, AtomicLong>();
        for (String counter : key.getComponentCounters()) {
            counters.put(counter, new AtomicLong(0));
        }
        stopwatchSamples = new ArrayList<AtomicLong>();
    }

    public void setSamples(int samples) {
        this.samples = samples;
    }

    public void increment(String counter) {
        if (counters.containsKey(counter)) {
            AtomicLong val = counters.get(counter);
            val.addAndGet(1);
        }
    }

    public void decrement(String counter) {
        if (!counters.containsKey(counter)) {
            logger.error("Unable to find counter "+ counter);
        } else {
            AtomicLong val = counters.get(counter);
            val.addAndGet(-1);
        }
    }

    private synchronized AtomicLong getAverageTime() {
        long totalTime = 0;
        for (AtomicLong time : stopwatchSamples) {
            totalTime += time.get();
        }
        return new AtomicLong(totalTime/stopwatchSamples.size());
    }

    private synchronized void addSample(AtomicLong sample) {
        if (stopwatchSamples.size() == samples) {
            stopwatchSamples.remove(0);
        }
        stopwatchSamples.add(sample);
    }

    public Chronometer createStopWatch(final String key) {

        return new Chronometer() {

            @Override
            public void stop() {
                AtomicLong elapsedTime = new AtomicLong(getTime());
                addSample(elapsedTime);
                counters.put(key, getAverageTime());
                reset();
            }
        };
    }

    @Override
    public Object getAttribute(String counter) throws AttributeNotFoundException,
                                                   MBeanException,
                                                   ReflectionException {
        return counters.get(counter);
    }

    @Override
    public AttributeList getAttributes(String[] keys) {
        AttributeList list = new AttributeList();
        for (String key : keys) {
            list.add(new Attribute(key, counters.get(key)));
        }
        return list;
    }

    @Override
    public MBeanInfo getMBeanInfo() {
        List<MBeanAttributeInfo> mbeanAttributes = new ArrayList<MBeanAttributeInfo>();
        for (Map.Entry<String, AtomicLong> entry : counters.entrySet()) {
            String id = entry.getKey();
            MBeanAttributeInfo mbai = new MBeanAttributeInfo(id,
                                                             entry.getValue().getClass().getName(),
                                                             id,
                                                             true,
                                                             false,
                                                             false);
            mbeanAttributes.add(mbai);
        }
        return new MBeanInfo(this.getClass().getName(),
                             key.getComponentId(),
                             mbeanAttributes.toArray(new MBeanAttributeInfo[]{}),
                             null,
                             null,
                             null);
    }

    /* (non-Javadoc)
     * @see javax.management.DynamicMBean#invoke(java.lang.String, java.lang.Object[], java.lang.String[])
     */
    @Override
    public Object
            invoke(String arg0, Object[] arg1, String[] arg2) throws MBeanException,
                                                             ReflectionException {
        return getAttributes(new String[]{});
    }

    /* (non-Javadoc)
     * @see javax.management.DynamicMBean#setAttribute(javax.management.Attribute)
     */
    @Override
    public void setAttribute(Attribute arg0) throws AttributeNotFoundException,
                                            InvalidAttributeValueException,
                                            MBeanException,
                                            ReflectionException {

    }

    /* (non-Javadoc)
     * @see javax.management.DynamicMBean#setAttributes(javax.management.AttributeList)
     */
    @Override
    public AttributeList setAttributes(AttributeList arg0) {
        // TODO Auto-generated method stub
        return new AttributeList();
    }

    private static ObjectName createObjectName(DiagnosticsManager bean)
            throws MalformedObjectNameException {
        String objectName = String.format(OBJECT_FORMAT,
                bean.key.getComponentId(), System.identityHashCode(bean));
        return new ObjectName(objectName);
    }

}
