/* **********************************************************
 * Copyright (c) 2015-2016, 2019 VMware, Inc. All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.client.rest;

import java.text.MessageFormat;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import com.vmware.vapi.CoreException;
import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.data.DataType;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.OptionalValue;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.internal.protocol.client.rpc.http.Utils;
import com.vmware.vapi.internal.protocol.common.http.UrlUtil;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.protocol.common.http.HttpConstants;

// TODO unit tests, logging and exception handling. Think about multi-threading.
/**
 * Builds HTTP request data of a vAPI operation based on an {@link OperationRestMetadata}. From the
 * vAPI operation's input ( {@link StructValue} and {@link ExecutionContext}) build HTTP request:
 * URL path, request body, HTTP headers.
 */
public class RestMetadataBasedHttpRequestBuilder {
    private final OperationRestMetadata restMetadata;
    private final UrlUtil urlUtils;

    /**
     * Constructs an HTTP request builder based on the provided {@link OperationRestMetadata}.
     *
     * @param restMetadata Metadata about the REST representation of an operation. Must not be
     *        <code>null</code>
     * @param urlUtil utility for dealing with URIs. Must not be <code>null</code>
     */
    public RestMetadataBasedHttpRequestBuilder(OperationRestMetadata restMetadata,
                                               UrlUtil urlUtil) {
        Validate.notNull(urlUtil);
        Validate.notNull(restMetadata);
        this.urlUtils = urlUtil;
        this.restMetadata = restMetadata;
    }

    /**
     * Build URL path of a HTTP request using the {@link OperationRestMetadata} provided to the
     * constructor and the vAPI operation {@link StructValue} <code>input</code>. The operation REST
     * metadata indicates how the path is to be customized by the <code>input</code> value for e.g.
     * path variables; query params etc.
     *
     * @param input value of the vAPI operation that customizes the URL path according to the
     *        operation REST metadata. Must not be <code>null</code>.
     * @return URL path customized by the <code>input</code> value.
     * @throws IllegalArgumentException if an URL path cannot be constructed with the provided
     *         <code>input</code> - the customization defined in the operation REST metadata cannot
     *         be fulfilled.
     */
    public String buildUrlPath(StructValue input) {
        Validate.notNull(input);
        Map<String, String> pathVariablesToFieldNames = restMetadata.getPathVariablesToFieldNames();
        Map<String, String> pathParams = extractParams(input,
                                                       pathVariablesToFieldNames,
                                                       false,
                                                       true);
        String path = urlUtils.replacePathVariables(restMetadata.getUrlTemplate(), pathParams);

        // Handle variable request/query params
        Map<String, String> requestParams = extractParams(input,
                                                          restMetadata
                                                                  .getRequestParamsToFieldNames(),
                                                          true,
                                                          false);
        String queryString = urlUtils.encodeQuery(requestParams);
        return Utils.appendQueryToPath(path, queryString);
    }

    public String buildAbsoluteTargetUrl(StructValue input, String baseUrl) {
        Validate.notNull(baseUrl);
        return urlUtils.joinUrls(baseUrl, buildUrlPath(input));
    }

    /**
     * Extract the request body {@link DataValue} from the operation's <code>input</code>.
     *
     * @param input value of the vAPI operation invocation. Must not be <code>null</code>
     * @return value of the request body or <code>null</code> if no request body is defined in the
     *         operation REST metadata
     * @throws IllegalArgumentException if the found request body is not of type {@link StructValue}
     * @throws CoreException if the <code>input</code> does not contain a request body field
     */
    public DataValue getRequestBodyValue(StructValue input) {
        Validate.notNull(input);
        if (restMetadata.getRequestBodyFieldName() != null) {
            return input.getField(restMetadata.getRequestBodyFieldName());
        } else {
            return null;
        }
    }

    /**
     * Build HTTP request headers from the operation's REST metadata and <code>input</code>.
     *
     * @param input value of the vAPI operation invocation. Must not be <code>null</code>.
     * @return a map of header name to list of header values to be added to the HTTP request.
     */
    public Map<String, List<String>> buildHeaders(StructValue input) {
        Validate.notNull(input);
        Map<String, List<String>> headers = new HashMap<>(restMetadata.getFixedHeaders());

        // Request header params
        Map<String, String> requestHeaders =
                extractParams(input,
                              restMetadata.getRequestHeadersToFieldNames(),
                              true,
                              false);
        for (Entry<String, String> header : requestHeaders.entrySet()) {
            Utils.addListEntryToMapOfLists(header.getKey(), header.getValue(), headers);
        }

        // contentType and accept attributes from RequestMapping
        String accept = restMetadata.getAccept();
        if (accept != null) {
            Utils.addListEntryToMapOfLists(HttpConstants.HEADER_ACCEPT, accept, headers);
        }
        String contentType = restMetadata.getContentType();
        if (contentType != null) {
            Utils.addListEntryToMapOfLists(HttpConstants.HEADER_CONTENT_TYPE, contentType, headers);
        }

        return headers;
    }

    private Map<String, String> extractParams(StructValue input,
                                              Map<String, String> paramsToFieldNames,
                                              boolean allowOptional,
                                              boolean urlEncodeValues) {
        Map<String, String> result = new HashMap<>();
        for (Entry<String, String> paramToFieldName : paramsToFieldNames.entrySet()) {
            String fieldName = paramToFieldName.getValue();
            if(!input.hasField(fieldName)) {
                continue;
            }
            DataValue fieldValue = input.getField(fieldName);
            fieldValue = unwrapOptionalValue(fieldValue);

            // Handle fieldValue null
            String name = paramToFieldName.getKey();
            if (fieldValue == null) {
                if(allowOptional){
                    continue;
                }else {
                    throw new IllegalArgumentException(MessageFormat.format("Field {0} cannot be null", name));
                }
            }

            switch (fieldValue.getType()) {
            // TODO Decide what to do with BINARY and SECRET types.
            case STRING:
            case BOOLEAN:
            case INTEGER:
                String value = fieldValue.toString();
                result.put(name,
                           urlEncodeValues ? urlUtils.encodeUrlPath(value) : value);
                break;
            case DOUBLE:
                // Double in RAML is defined by YAML 1.2 spec
                // See YAML 1.2 section 10.2.1.4 Floating Point
                // http://yaml.org/spec/1.2/spec.html#id2804092
                // Re: -? [1-9] ( \. [0-9]* [1-9] )? ( e [-+] [1-9] [0-9]*
                // )?.
                // For now we do not support it
            default:
                String msg = String
                        .format("Unsupported type of value of path, query or "
                                + "header variable: %s . "
                                + "Only String, Boolean and Integer are supported.", fieldValue
                                .getType().toString());
                throw new IllegalArgumentException(msg);
            }
        }
        return result;
    }

    private static DataValue unwrapOptionalValue(DataValue value) {
        if (value != null && value.getType() == DataType.OPTIONAL) {
            return unwrapOptionalValue(((OptionalValue) value).getValue());
        } else {
            return value;
        }
    }

}
