/* **********************************************************
 * Copyright (c) 2016 VMware, Inc. All rights reserved. VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.client.rest.authn;

import java.nio.charset.StandardCharsets;

import org.apache.commons.codec.binary.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.ExecutionContext.SecurityContext;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.internal.protocol.client.rpc.HttpRequest;
import com.vmware.vapi.protocol.client.http.RequestPreProcessor;
import com.vmware.vapi.security.StdSecuritySchemes;
import com.vmware.vapi.security.UserPassSecurityContext;

/**
 * Request preprocessor class that appends HTTP Basic Authorization header to an HTTP request.
 */
public class BasicAuthenticationAppender implements RequestPreProcessor {
    private static final Logger logger = LoggerFactory.getLogger(BasicAuthenticationAppender.class);

    /**
     * Extracts username and password from the provided {@link ExecutionContext} and appends an HTTP
     * Basic Authorization header to the request. While base64 encoding the method assumes that
     * both username and password are "UTF_8" encoded strings (see RFC-2617 and RFC-2616). If
     * username and password cannot be extracted from the {@link ExecutionContext} the method does
     * not mutate the provided {@link HttpRequest} and returns it unchanged.
     * <br/>
     * <p>
     * {@inheritDoc}
     */
    @Override
    public HttpRequest handle(String serviceId,
                              String operationId,
                              HttpRequest request,
                              DataValue params,
                              ExecutionContext context) {
        logger.trace("Trying to append HTTP Basic Authorization header to request: {} {}",
                     request.getUrl(),
                     request.getMethod());
        SecurityContext securityContext = context.retrieveSecurityContext();

        if (securityContext == null) {
            logger.debug("No security context available - "
                                 + "HTTP Basic Authorization header will not be added to request: {} {}",
                         request.getUrl(),
                         request.getMethod());
            return request;
        }

        if (logger.isTraceEnabled()) {
            logger.trace("SecurityContext properties: {}", securityContext.getAllProperties()
                    .keySet());
        }

        Object authnSchemeId = securityContext
                .getProperty(SecurityContext.AUTHENTICATION_SCHEME_ID);
        logger.trace("Authentication scheme is {}.", authnSchemeId);
        if (!StdSecuritySchemes.USER_PASS.equals(authnSchemeId)) {
            logger.debug("Security scheme is not User Pass - "
                                 + "HTTP Basic Authorization header will not be added to request: {} {}",
                         request.getUrl(),
                         request.getMethod());
            return request;
        }

        String username = (String) securityContext.getProperty(UserPassSecurityContext.USER_KEY);
        char[] password = (char[]) securityContext
                .getProperty(UserPassSecurityContext.PASSWORD_KEY);
        // TODO: Cleaner way to implement the previous 2 lines is to use the SCF as below.
        // However it is in vapi-authn module, and adding dependency to this here in
        // vapi-rest is problematic because of the vapi-auth vs vapi-authentication.
        // More precisely adding the dep here breaks vapi-rest-authentication, because
        // it gets both vapi-auth and vapi-authentication in its transitive closure of
        // dependencies. We have at least 2 options:
        // - move this class in vapi-rest-authn
        // - move the SCF logic we need here from vapi-authn to vapi-runtime (be careful
        // not to move SAML token stuff, they must stay in vapi-authn; so it is not
        // just a straight class movement)
        //
//        UserPassSecurityContext userPassCtx = SecurityContextFactory
//                .parseUserPassSecurityContext(securityContext);
//        String username = userPassCtx.getUser();
//        char[] password = userPassCtx.getPassword();

        if (username == null || password == null) {
            logger.warn("Either username or password is missing from security context - "
                                + "HTTP Basic Authorization header will not be added to request: {} {}",
                        request.getUrl(),
                        request.getMethod());
            return request;
        }
        return addAuthorizationHeader(username, password, request);
    }

    private static HttpRequest addAuthorizationHeader(String username,
                                                      char[] password,
                                                      HttpRequest request) {
        String authString = username + ":" + new String(password);
        String authEncodedString = Base64.encodeBase64String(authString
                .getBytes(StandardCharsets.UTF_8));
        String headerValue = String.format("Basic %s", authEncodedString);
        request.addHeader("Authorization", headerValue);

        logger.debug("HTTP Basic Authorization header appended successfully to request: {} {}.",
                     request.getUrl(),
                     request.getMethod());
        return request;
    }
}
