/* **********************************************************
 * Copyright (c) 2012-2014, 2018, 2020, 2022 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.protocol.client.rpc;

import java.io.InputStream;
import java.util.Collection;
import java.util.Objects;

import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.internal.core.abort.AbortHandle;
import com.vmware.vapi.internal.tracing.TracingSpan;

/**
 * <p>
 * Client transport which supports correlation between requests and responses.
 * The implementation does not have to be asynchronous. The
 * {@link #send(InputStream, int, AbortHandle, ResponseCallback)} method might
 * block until all response frames are received and invoke the response callback
 * synchronously.
 * </p>
 * <p>
 * Each request must have at least one response frame, otherwise the transport
 * has to report an error through the response callback.
 * </p>
 */
public interface CorrelatingClient {

    /*
     * If there comes a need to have a non-correlating transport (e.g. full
     * duplex frame transport over a single TCP connection), create another
     * abstraction (e.g. NonCorrelatingAsyncClient), do not use this one.
     */

    /**
     * Interface for controlling the transport. This is useful to manage back
     * pressure i.e. if the client cannot process more data it can defer reading
     * the response and thus exert pressure using the underlying transport
     * mechanism e.g. TCP.
     * <p>
     * It is important to note that this is rather coarse control dependent upon
     * the transport mechanism. API designers may want to expose their own flow
     * control & acknowledgement APIs for finer grained and more time sensitive
     * controls.
     * <p>
     * Suspending read may not take immediate effect. Callers must be able to
     * handle frames after invoking {@link #suspendRead()}.
     */
    interface TransportControl {
        /**
         * Suspend reading of transport frames.
         */
        void suspendRead();

        /**
         * Resume reading of transport frames.
         */
        void resumeRead();

        /**
         * Abort reading of transport frames.
         */
        void cancel();
    }

    /**
     * Special instance to be sent to {@link ResponseCallback} implementations
     * when there is no meaningful control option. For example single frame
     * transports can use this. The goal of this is to avoid sending
     * {@code null} control instance.
     */
    public static TransportControl NO_OP_CONTROL = new TransportControl() {
        @Override
        public void suspendRead() {
            // do nothing
        }

        @Override
        public void resumeRead() {
            // do nothing
        }

        @Override
        public void cancel() {
            // do nothing
        }
    };

    /**
     * Callback used to receive response frames for a given request frame.
     * Errors are also reported through this callback.
     */
    interface ResponseCallback {
        /**
         * Notifies the callback that an error occurred while sending the
         * request or while receiving a response.
         *
         * @param error transport error; must not be <code>null</code>
         */
        void failed(RuntimeException error);

        /**
         * <p>
         * Notifies the callback that a response frame is received. Can be
         * invoked multiple times - once for each response frame. All responses
         * will be correlated to a single request.
         * </p>
         * <p>
         * If an exception is thrown (i.e. the response frame is considered
         * invalid for some reason), the transport session would be terminated
         * and no more response frames would be received for the request. The
         * exception would trigger the {@link #failed(RuntimeException)} method.
         * </p>
         *
         * @param response response frame
         * @param control transport control interface. Must not be {@code null}
         */
        void received(InputStream response, TransportControl control);

        /**
         * Signals stream completion to the upper layer. After this signal is
         * fired the {@link ResponseCallback} will no longer be called. This
         * signal may not be sent on specific transport e.g. message bus. It is
         * expected that the upper layer uses this signal to validate the stream
         * state i.e. that upon stream completion the client expects no more
         * frames.
         */
        void completed();
    }

    interface ResponseCallbackFactory {
        /**
         * Creates response callback for particular API response. For example
         * based on the content type header different de-serialization strategy
         * may be chosen.
         *
         * @param prams response description
         * @return instance for de-serializing the particular response.
         */
        ResponseCallback createResponseCallback(ResponseCallbackParams params);

        /**
         * Notifies that an error occurred before receiving a response.
         *
         * @param error transport error; must not be <code>null</code>
         */
        void failed(RuntimeException error);
    }

    /**
     * Parameters for creation of a {@link ResponseCallback} for particular
     * response. This matches the response metadata to appropriate response
     * callback. For example it could match the {@code content-type} header to
     * de-serialize different response types like clean JSON vs. JSON with type
     * information .
     */
    public static class ResponseCallbackParams {
        private String contentType;

        public ResponseCallbackParams(String contentType) {
            this.contentType = contentType;
        }

        /**
         * Returns the MIME type of the response.
         */
        public String getContentType() {
            return contentType;
        }
    }

    /**
     * Request parameters object
     */
    public static class SendParams {
        private final InputStream request;
        private final int requestLength;
        private ExecutionContext executionContext;
        private AbortHandle abortHandle;
        private final ResponseCallbackFactory cbFactory;
        private final String serviceId;
        private final String operationId;
        private final String contentType;
        private final Collection<String> acceptedTypes;
        private final TracingSpan tracingSpan;

        /**
         * Constructs a request parameters object.
         *
         * @param serviceId service identifier; must not be {@code null}
         * @param operationId operation identifier; must not be {@code null}
         * @param request request frame
         * @param requestLength length of the request as number of bytes; this
         *        is just a hint to the implementation, pass <code>-1</code> if
         *        the length is unknown
         * @param executionContext carries information auxiliary to the request
         *        which might not be represented on the wire
         * @param cbFactory response callback factory; creates the
         *        {@link ResponseCallback}; must not be {@code null}
         * @param abortHandle abort handle used to optionally abort the request
         *        execution; can be null
         * @param contentType content type of the request; must not be
         *        {@code null}
         * @param acceptedTypes the content types apparent in the {@code Accept}
         *        header of the request; must not be {@code null}
         * @param tracingSpan the span containing tracing information about the
         *        current operation
         */
        public SendParams(String serviceId,
                          String operationId,
                          InputStream request,
                          int requestLength,
                          ExecutionContext executionContext,
                          ResponseCallbackFactory cbFactory,
                          AbortHandle abortHandle,
                          String contentType,
                          Collection<String> acceptedTypes,
                          TracingSpan tracingSpan) {
            Objects.requireNonNull(serviceId);
            Objects.requireNonNull(operationId);
            Objects.requireNonNull(request);
            Objects.requireNonNull(executionContext);
            Objects.requireNonNull(cbFactory);
            Objects.requireNonNull(contentType);
            Objects.requireNonNull(acceptedTypes);
            Objects.requireNonNull(tracingSpan);
            if (requestLength < -1) {
                throw new IllegalArgumentException("Invalid request length.");
            }

            this.serviceId = serviceId;
            this.operationId = operationId;
            this.request = request;
            this.requestLength = requestLength;
            this.executionContext = executionContext;
            this.cbFactory = cbFactory;
            this.abortHandle = abortHandle;
            this.contentType = contentType;
            this.acceptedTypes = acceptedTypes;
            this.tracingSpan = tracingSpan;
        }

        public InputStream getRequest() {
            return request;
        }

        public int getRequestLength() {
            return requestLength;
        }

        public ExecutionContext getExecutionContext() {
            return executionContext;
        }

        public void setExecutionContext(ExecutionContext executionContext) {
            this.executionContext = executionContext;
        }

        public AbortHandle getAbortHandle() {
            return abortHandle;
        }

        public void setAbortHandle(AbortHandle abortHandle) {
            this.abortHandle = abortHandle;
        }

        public ResponseCallbackFactory getCbFactory() {
            return cbFactory;
        }

        public String getServiceId() {
            return serviceId;
        }

        public String getOperationId() {
            return operationId;
        }

        public String getContentType() {
            return contentType;
        }

        public Collection<String> getAcceptedTypes() {
            return acceptedTypes;
        }

        public TracingSpan getTracingSpan() {
            return tracingSpan;
        }
    }

    /**
     * Sends a request and registers a callback to receive the response. Each
     * request can have one or more responses.
     *
     * @param params request parameters
     */
    void send(SendParams params);

    /**
     * Closes the transport.
     */
    void close();
}
