/* **********************************************************
 * Copyright (c) 2019, 2022 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.client.rpc.http;

import java.lang.ref.WeakReference;
import java.util.Iterator;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A singleton class that deals with cleaning up expired connections in
 * registered connection pools.
 */
public final class ConnectionMonitor {
    private static final Logger LOG = LoggerFactory
            .getLogger(ConnectionMonitor.class);
    public static final String THREAD_NAME_PROP =
            "com.vmware.vapi.internal.protocol.client.http.cleanupThreadName";
    public static final String DEFAULT_THREAD_NAME = "vAPI-client-connection-monitor";
    public static final String CLEANUP_PERIOD_IN_SECONDS_PROP =
            "com.vmware.vapi.internal.protocol.client.http.cleanupPeriod";
    public static final int DEFAULT_CLEANUP_PERIOD_IN_SECONDS = 30;
    public static final String KEEP_ALIVE_IN_SECONDS_PROP =
            "com.vmware.vapi.internal.protocol.client.http.keepAlive";
    public static final int DEFAULT_KEEP_ALIVE_IN_SECONDS = 45;

    private static volatile boolean initialized;
    private static final ScheduledThreadPoolExecutor executor;
    private static final ConcurrentLinkedQueue<WeakReference<CleanableConnectionPool>> connectionPools;
    private static ScheduledFuture<?> task;

    private ConnectionMonitor() { }

    static {
        initialized = false;
        executor = new ScheduledThreadPoolExecutor(1, new ThreadFactory() {
            @Override
            public Thread newThread(Runnable r) {
                String name = System.getProperty(THREAD_NAME_PROP, DEFAULT_THREAD_NAME);
                if (LOG.isTraceEnabled()) {
                    LOG.debug(String.format("Starting thread %s to monitor for "
                                            + "expired connections",
                                            name));
                }
                Thread t = new Thread(r, name);
                t.setDaemon(true);
                return t;
            }
        });

        Integer keepAlive = Integer.getInteger(KEEP_ALIVE_IN_SECONDS_PROP, DEFAULT_KEEP_ALIVE_IN_SECONDS);
        executor.setKeepAliveTime(keepAlive, TimeUnit.SECONDS);
        executor.allowCoreThreadTimeOut(true);
        connectionPools = new ConcurrentLinkedQueue<WeakReference<CleanableConnectionPool>>();
    }

    /**
     * Registers the specified connection pool for monitoring.
     * <p>
     * This method uses a weak-reference to keep track of the specified
     * connection pool, so that it does not prevent it being garbage collected.
     * <p>
     * Once registered, the connection pool would get its
     * {@link CleanableConnectionPool#closeExpiredConnections()
     * closeExpiredConnections} method invoked every 10 seconds.
     * <p>
     * This method is thread-safe.
     *
     * @param pool the connection pool; must not be {@code null}
     */
    public static void register(CleanableConnectionPool pool) {
        if (pool == null) {
            throw new NullPointerException("pool cannot be null");
        }

        if (LOG.isDebugEnabled()) {
            LOG.debug(String.format("A connection pool (%h) is being added for "
                                    + "monitoring",
                                    pool));
        }

        synchronized (connectionPools) {
            connectionPools
                    .add(new WeakReference<CleanableConnectionPool>(pool));
            initialize();
        }
    }

    private static void initialize() {
        if (initialized) {
            return;
        }

        LOG.debug("Starting to monitor connection pools for expired connections");
        Integer period = Integer.getInteger(CLEANUP_PERIOD_IN_SECONDS_PROP, DEFAULT_CLEANUP_PERIOD_IN_SECONDS);
        task = executor.scheduleAtFixedRate(new ConnectionCleaner(),
                                            period,
                                            period,
                                            TimeUnit.SECONDS);
        initialized = true;
    }

    /**
     * Stops the scheduled execution of the clean-up task and clears all
     * registered connection pools.
     * <p>
     * Available for testing purposes only. Not thread-safe.
     */
    static void reset() {
        synchronized (connectionPools) {
            connectionPools.clear();
            stop();
        }
    }

    private static void stop() {
        if (!initialized || !connectionPools.isEmpty()) {
            return;
        }

        LOG.trace("Stopping connection monitoring");
        initialized = false;
        task.cancel(false);
        executor.getQueue().remove(task);
        task = null;
    }

    private static class ConnectionCleaner implements Runnable {
        @Override
        public void run() {
            try {
                LOG.trace("Starting to clean-up expired vAPI-client connections");
                int cleanedUp = closeConnections();
                if (LOG.isDebugEnabled()) {
                    LOG.debug(String.format("Cleaned-up %s connection pool(s)",
                                            cleanedUp));
                }

                if (cleanedUp == 0) {
                    synchronized (connectionPools) {
                        stop();
                    }
                }
            } catch (RuntimeException e) {
                // Shouldn't come here, but still handled to make sure the
                // task doesn't get dropped by the executor upon exception
                LOG.error("An exception occured while iterating over the"
                          + "vAPI-client connection pools",
                          e);
            }
        }

        /**
         * @return number of cleaned pools
         */
        private int closeConnections() {
            int i = 0;
            int cleanedUp = 0;
            Iterator<WeakReference<CleanableConnectionPool>> cpIterator = connectionPools
                    .iterator();
            while (cpIterator.hasNext()
                   && !Thread.currentThread().isInterrupted()) {
                i++;
                WeakReference<CleanableConnectionPool> cpRef = cpIterator
                        .next();
                CleanableConnectionPool cp = cpRef.get();
                if (cp == null) {
                    if (LOG.isTraceEnabled()) {
                        LOG.trace(String
                                .format("Removing GCed connection pool at "
                                        + "position %s of the pools queue",
                                        i - 1));
                    }
                    cpIterator.remove();
                    continue;
                }

                if (LOG.isTraceEnabled()) {
                    LOG.trace(String
                            .format("Closing expired connections for pool %h",
                                    cp));
                }
                try {
                    cp.closeExpiredConnections();
                } catch (RuntimeException e) {
                    if (LOG.isWarnEnabled()) {
                        LOG.warn(String
                                .format("Unable to clean-up expired connections"
                                        + " for pool %h",
                                        cp),
                                 e);
                    }
                }
                cleanedUp = cleanedUp + 1;
            }

            return cleanedUp;
        }
    }

    /**
     * A connection pool that needs to be monitored for expired connections.
     */
    public interface CleanableConnectionPool {
        /**
         * Closes the connections that have stayed available without being
         * re-used for more time then their keep-alive period.
         */
        void closeExpiredConnections();

        /**
         * This method is available for test purposes and usually serves to
         * verify that connections are not leaking.
         *
         * @return the number of connections in the pool that are currently
         *         being used; does not include the connections that are being
         *         kept alive for future use.
         */
        int leasedConnections();
    }
}