/* **********************************************************
 * Copyright (c) 2012-2015, 2018-2022 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.client.rpc.http;

import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.protocol.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.internal.core.abort.AbortHandle;
import com.vmware.vapi.internal.protocol.client.rpc.CorrelatingClient;
import com.vmware.vapi.internal.protocol.client.rpc.http.ConnectionMonitor.CleanableConnectionPool;
import com.vmware.vapi.internal.protocol.client.rpc.http.handle.BioResponseContentHandler;
import com.vmware.vapi.internal.protocol.common.Util;
import com.vmware.vapi.internal.protocol.common.http.ApacheHttpClientExceptionTranslator;
import com.vmware.vapi.internal.tracing.TracingSpan;
import com.vmware.vapi.internal.tracing.otel.TracingAttributeKey.IoType;
import com.vmware.vapi.internal.util.TracingUtil;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.protocol.HttpConfiguration;
import com.vmware.vapi.protocol.HttpConfiguration.Protocol;

/**
 * This class is vAPI transport implementation that uses Apache
 * {@link org.apache.http.client.HttpClient}.
 */
public final class HttpClient implements CorrelatingClient {
    private static Logger logger = LoggerFactory.getLogger(HttpClient.class);

    private final String uri;
    private final CloseableHttpClient httpClient;
    private final ApacheClientRequestConfigurationMerger configMerger;
    private final Protocol protocol;

    // Hard link to connection pool used for monitoring client connections
    private CleanableConnectionPool pool;

    /**
     * Creates a HTTP client using default configuration.
     *
     * @param uri  URI representing the server to connect. Should be of the form
     *             <tt>http(s)://[IP_address]:[port]/[Service_endpoint]</tt>. Can
     *             not be null.
     */
    public HttpClient(String uri) {
        this(uri, new HttpConfiguration.Builder().getConfig());
    }

    /**
     * Creates a HTTP client using user specified configuration.
     *
     * @param uri  URI representing the server to connect. Should be of the form
     *             <tt>http(s)://[IP_address]:[port]/[Service_endpoint]</tt>. Can
     *             not be null.
     * @param httpConfig HTTP client configuration. Can not be null.
     */
    public HttpClient(String uri,
                      HttpConfiguration httpConfig) {
        // TODO - validate that URI has port set
        Validate.notNull(uri);
        Validate.notNull(httpConfig);
        this.uri = uri;
        this.protocol = httpConfig.getProtocol();
        this.httpClient = createDefaultHttpClient(httpConfig);
        RequestConfig defaultConfig = ApacheHttpUtil.createDefaultRequestConfig(httpConfig);
        this.configMerger = new ApacheClientRequestConfigurationMerger(defaultConfig);
    }

    /**
     * Creates a HTTP client using user specified configuration.
     *
     * @param uri  URI representing the server to connect. Should be of the form
     *             <tt>http(s)://[IP_address]:[port]/[Service_endpoint]</tt>. Can
     *             not be null.
     * @param httpClient the HTTP client that will be used as a transport.
     *                   Can not be null. WARNING - calling close() will
     *                   shutdown the connection manager of the HTTP client.
     */
    public HttpClient(String uri,
                      CloseableHttpClient httpClient) {
        // TODO - validate that URI has port set
        Validate.notNull(uri);
        Validate.notNull(httpClient);
        this.uri = uri;
        this.httpClient = httpClient;
        this.protocol = Protocol.VAPI_JSON_RPC_1_0;
        this.configMerger = new ApacheClientRequestConfigurationMerger(null);
    }

    /**
     * Creates a default Apache HttpClient using the provided configuration
     *
     * @param httpConfig cannot be null.
     * @return
     */
    private CloseableHttpClient createDefaultHttpClient(
            final HttpConfiguration httpConfig) {

        assert httpConfig != null;

        // validate the uri while cleating the client instance
        try {
            new URL(uri);
        } catch (MalformedURLException ex) {
            logger.error("Exception while trying to parse URL", ex);
            throw new RuntimeException(ex);
        }

        ApacheBioHttpClientBuilder httpClientBuilder = new ApacheBioHttpClientBuilder();
        CloseableHttpClient result = httpClientBuilder.buildAndConfigure(httpConfig);

        // register client with a thread which will periodically clean up
        // connections which are closed by the other end, or expired from
        // the connection pool; this is recommended way to improve resource
        // utilization since the pool will only do this clean up on
        // connection lease requests
        pool = httpClientBuilder.registerClientWithConnectionMonitor();

        return result;
    }

    private HttpResponse invoke(SendParams params)
            throws IOException {
        final HttpPost post = new HttpPost(uri);
        post.setEntity(new InputStreamEntity(params.getRequest(),
                                             params.getRequestLength()));

        Util.addHeaders(post,
                        params.getContentType(),
                        params.getAcceptedTypes(),
                        protocol,
                        params.getServiceId(),
                        params.getOperationId(),
                        params.getExecutionContext());
        logger.debug("Value of uri is:" + uri);

        if (!Util.checkRequestAborted(params.getAbortHandle(),
                                      params.getCbFactory())) {

            Util.registerAbortListerner(params.getCbFactory(),
                                        post,
                                        params.getAbortHandle());
        } else {
            // Request aborted, no need to continue
            return null;
        }

        HttpContext context = ApacheHttpUtil
                .createHttpContext(params.getExecutionContext(), configMerger);

        TracingSpan tracingSpan = params.getTracingSpan();
        TracingUtil.registerRequestDataIntoSpan(tracingSpan, post, protocol, IoType.BIO);
        tracingSpan.injectInto(post, TracingUtil::addHeader);

        HttpResponse response = httpClient.execute(post, context);
        TracingUtil.registerResponseIntoSpan(tracingSpan, response, context);

        ApacheHttpUtil.validateHttpResponse(response, params.getAcceptedTypes());

        return response;
    }

    @Override
    public void send(SendParams params) {
        if (Util.checkRequestAborted(params.getAbortHandle(), params.getCbFactory())) {
            return;
        }

        ResponseCallbackFactory cbFactory = params.getCbFactory();
        try {
            HttpResponse httpResponse = invoke(params);
            HttpEntity responseBody = httpResponse.getEntity();
            if (responseBody == null) {
                // Failure due to abort, error is already set, just return
                return;
            }
            String contentType = null;
            if (responseBody.getContentType() != null) {
                contentType = responseBody.getContentType().getValue();
            }

            ResponseCallbackParams cbParams = new ResponseCallbackParams(contentType);
            ResponseCallback cb = cbFactory.createResponseCallback(cbParams);

            InputStream inp = responseBody.getContent();
            handleContent(contentType, inp, cb, params.getAbortHandle(), uri);

        } catch (Exception ex) {
            cbFactory.failed(ApacheHttpClientExceptionTranslator.translate(ex,
                                                                    params.getAbortHandle(),
                                                                    uri));
        }
    }

    /**
     * Processes an HTTP response content.
     *
     * @param contentType value of Content-Type header; may be <code>null</code>
     * @param content HTTP response content
     * @param cb response callback
     * @param abortHandle abort handle
     * @param uri address for debug log
     * @throws IOException I/O error
     */
    private void handleContent(String contentType,
                               InputStream content,
                               ResponseCallback cb,
                               AbortHandle abortHandle,
                               String uri)
            throws IOException {
        BioResponseContentHandler contentHandler = new BioResponseContentHandler(contentType,
                                                                                 cb);
        contentHandler.handleResposne(content, abortHandle, uri);
    }

    @Override
    public void close() {
        try {
            httpClient.close();
            pool = null;
        } catch (IOException e) {
            logger.warn("Exception while trying to close the HttpClient", e);
        }
    }
}
