/* **********************************************************
 * Copyright (c) 2015-2016, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.client.rpc.http;

import java.util.Collections;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.MethodResult;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.internal.protocol.client.rpc.HttpRequest;
import com.vmware.vapi.internal.protocol.client.rpc.HttpResponse;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.protocol.client.http.RequestPreProcessor;
import com.vmware.vapi.protocol.client.http.ResponsePostProcessor;

/**
 * Aggregate multiple request/response processors. Execute them in the order
 * they were registered.
 */
// TODO name? aggregator?
// TODO error handling? should there be error handling in the post/pre
// processors or the one invoking them should be responsible
public class ProcessorManager implements ResponsePostProcessor, RequestPreProcessor {
    private static final Logger LOGGER = LoggerFactory
            .getLogger(ProcessorManager.class);

    private List<RequestPreProcessor> preProcessors = Collections.emptyList();
    private List<ResponsePostProcessor> postProcessors = Collections.emptyList();

    public void setPreProcessors(List<RequestPreProcessor> processors) {
        Validate.notNull(processors);
        this.preProcessors = processors;
    }

    public void setPostProcessors(List<ResponsePostProcessor> processors) {
        Validate.notNull(processors);
        this.postProcessors = processors;
    }

    @Override
    public HttpRequest handle(String serviceId,
                              String operationId,
                              HttpRequest request,
                              DataValue params,
                              ExecutionContext executionContext) {
        for (RequestPreProcessor processor : preProcessors) {
            try {
                request = processor.handle(serviceId, operationId, request, params, executionContext);
            } catch (Exception ex) {
                if (LOGGER.isErrorEnabled()) {
                    LOGGER.error("ResponsePostProcessor " + processor + " threw an exception", ex);
                }
            }
        }

        return request;
    }

    @Override
    public MethodResult handle(String serviceId,
                               String operationId,
                               MethodResult result,
                               HttpResponse response) {
        for (ResponsePostProcessor processor : postProcessors) {
            try {
                result = processor.handle(serviceId, operationId, result, response);
            } catch (Exception ex) {
                if (LOGGER.isErrorEnabled()) {
                    LOGGER.error("ResponsePostProcessor " + processor + " threw an exception", ex);
                }
            }
        }

        return result;
    }
}