/* **********************************************************
 * Copyright (c) 2013-2014, 2017, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.client.rpc.http;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.security.InvalidAlgorithmParameterException;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.PKIXBuilderParameters;
import java.security.cert.PKIXParameters;
import javax.net.ssl.CertPathTrustManagerParameters;
import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509KeyManager;

import com.vmware.vapi.protocol.HttpConfiguration;
import com.vmware.vapi.protocol.HttpConfiguration.SslConfiguration;
import com.vmware.vapi.client.exception.SslException;
import com.vmware.vapi.internal.util.Validate;

/**
 * SSL utility for HTTP clients.
 */
public final class SslClientUtil {

    private SslClientUtil() {
    }

    /**
     * Creates an SSL context for the specified configuration.
     *
     * @param sslConfig SSL configuration; must not be <code>null</code>
     * @return SSL context
     */
    public static SSLContext createSslContext(SslConfiguration sslConfig) {
        Validate.notNull(sslConfig);
        try {
            SSLContext sslCtx = SSLContext.getInstance("TLS");

            // Load the key managers that will provide the SSL connection
            // authentication keys.
            KeyManager[] keyManagers = createKeyManagers(sslConfig);
            TrustManager[] trustManagers = null;

            if (sslConfig.isCertificateValidationDisabled()) {
                trustManagers = new TrustManager[] {
                                               new TrustAllX509TrustManager() };
            } else {
                String trustAlgorithm = TrustManagerFactory.getDefaultAlgorithm();
                TrustManagerFactory trustFactory =
                        TrustManagerFactory.getInstance(trustAlgorithm);

                if (sslConfig.getCrlCertStore() == null) {
                    trustFactory.init(sslConfig.getTrustStore());
                } else {
                    PKIXParameters pkixParams = new PKIXBuilderParameters(
                                                      sslConfig.getTrustStore(),
                                                      null);
                    pkixParams.setRevocationEnabled(true);
                    pkixParams.addCertStore(sslConfig.getCrlCertStore());
                    trustFactory.init(new CertPathTrustManagerParameters(
                                                                   pkixParams));
                }

                trustManagers = trustFactory.getTrustManagers();
            }
            sslCtx.init(keyManagers, trustManagers, null);
            return sslCtx;
        } catch (NoSuchAlgorithmException ex) {
            throw new SslException(ex);
        } catch (KeyStoreException ex) {
            throw new SslException(ex);
        } catch (KeyManagementException ex) {
            throw new SslException(ex);
        } catch (InvalidAlgorithmParameterException ex) {
            throw new SslException(ex);
        }
    }

    /**
     * Create the key managers that will provide the SSL connection
     * authentication keys.
     *
     * @param config the {@link HttpConfiguration}
     * @return the key managers, or null if no authentication configuration
     *         is provided
     */
    protected static KeyManager[] createKeyManagers(SslConfiguration sslConfig) {
        if (sslConfig.getKeyStore() == null
            || sslConfig.getKeyStoreConfig() == null) {
            return null;
        }
        String password = sslConfig.getKeyStoreConfig().getKeyStorePassword();
        String alias = sslConfig.getKeyStoreConfig().getKeyAlias();

        try {
            if (password != null && alias != null) {
                KeyStore store = sslConfig.getKeyStore();
                KeyManagerFactory keyMngrFactory = KeyManagerFactory
                        .getInstance(KeyManagerFactory.getDefaultAlgorithm());
                keyMngrFactory.init(store, password.toCharArray());
                KeyManager[] keyManagers = keyMngrFactory.getKeyManagers();

                for (int i = 0; i < keyManagers.length; ++i) {
                    if (keyManagers[i] instanceof X509KeyManager) {
                        keyManagers[i] =
                                new KeyManagerWrapper((X509KeyManager) keyManagers[i],
                                                      alias);
                    }
                }

                return keyManagers;
            }
        } catch (Exception e) {
            throw new SslException("Unable to create KeyManagers", e);
        }

        return null;
    }

    static String stripHostnameBrackets(String hostname) {
        if ((hostname != null) && hostname.startsWith("[")
                && hostname.endsWith("]")) {

            try {
                // we are not afraid that hosname direct resolve will happen here
                // because the [] are only valid in IPv6 literals (and not
                // host names
                return InetAddress.getByName(hostname).getHostAddress();
            } catch (UnknownHostException e) {
               // if InetAddress bails, this is not, in fact a valid IPv6
               // address, so return it as is and let the flow go forward
            }

            // TODO: the above handling is to make sure that IPv6 address is
            //       normalized before the hostname verification.
            //       Such functionality is not available in httpclient 4.2.x,
            //       but this is fixed in 4.3.x. So do it here in order to
            //       keep the support for 4.2.x.
            //
            // this is what we want (to replace the try{} block above), after
            // all consumers are on httpclient 4.3.x:
            //
            // hostname = hostname.substring(1, hostname.length() - 1);
        }

        return hostname;
    }

    @SuppressWarnings("deprecation")
    static org.apache.http.conn.ssl.X509HostnameVerifier
                createHostnameVerifier(
                       boolean isHostnameVerificationDisabled,
                       org.apache.http.conn.ssl.X509HostnameVerifier delegate) {
        // TODO in version 4.4 Apache has introduced DefaultHostnameVerifier
        // that should replace the
        // SSLIOSessionStrategy.STRICT_HOSTNAME_VERIFIER. However, there are
        // still consumers at version older than 4.4.x and this is not available
        // for them. So we'll stick with the deprecated one till all consumers
        // switch to 4.4 or newer Apache HTTP Client.
        return isHostnameVerificationDisabled
                ? new AllowAllHostnameVerifier()
                : new VapiHostnameVerifier(delegate);
    }
}
