/* **********************************************************
 * Copyright (c) 2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.protocol.client.rpc.http.handle;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.nio.ByteBuffer;

import org.apache.http.HttpResponse;
import org.apache.http.nio.IOControl;
import org.apache.http.nio.client.methods.AsyncByteConsumer;
import org.apache.http.protocol.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.internal.protocol.client.rpc.CorrelatingClient;
import com.vmware.vapi.internal.protocol.client.rpc.CorrelatingClient.ResponseCallback;
import com.vmware.vapi.internal.protocol.common.http.BinaryInput;
import com.vmware.vapi.internal.protocol.common.http.FrameDeserializer;
import com.vmware.vapi.internal.protocol.common.http.impl.ByteBufferBinaryInput;

/**
 * Consumer which treats the response HTTP entity as a stream of frames.
 */
public class NioMultiResponseConsumer extends AsyncByteConsumer<HttpResponse> {

    private volatile HttpResponse httpResponse;
    private final ResponseCallback callback;
    private final FrameDeserializer deserializer;

    private static Logger logger = LoggerFactory
            .getLogger(NioMultiResponseConsumer.class);

    public NioMultiResponseConsumer(FrameDeserializer deserializer,
                                    ResponseCallback callback) {
        this.deserializer = deserializer;
        this.callback = callback;
    }

    @Override
    protected void onResponseReceived(final HttpResponse response) {
        httpResponse = response;
    }

    @Override
    protected void onByteReceived(final ByteBuffer buf, final IOControl ioctrl)
            throws IOException {
        BinaryInput input = new ByteBufferBinaryInput(buf);
        byte[] frame;

        while ((frame = readFrame(input)) != null) {
            if (logger.isDebugEnabled()) {
                logger.debug("Received response frame with size "
                             + frame.length);
            }
            callback.received(new ByteArrayInputStream(frame),
                              CorrelatingClient.NO_OP_CONTROL);
        }
    }

    @Override
    protected HttpResponse buildResult(final HttpContext context) {
        logger.debug("Streaming HTTP response complete");
        return httpResponse;
    }

    private byte[] readFrame(BinaryInput input) throws IOException {
        synchronized (deserializer) {
            return deserializer.readFrame(input);
        }
    }
}
