/* **********************************************************
 * Copyright 2013 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.protocol.common.http;

import java.io.IOException;

/**
 * <p>
 * Handle to an input source of binary data. The handle could represent the
 * whole input source or just a piece of it (i.e. the input source could be
 * represented by a sequence of input handles).
 * </p>
 * <p>
 * There are two final states of an input handle - {@link #NO_DATA_TEMPORARY}
 * and {@link #NO_DATA_PERMANENT}. {@link #NO_DATA_TEMPORARY} means that the
 * current handle is exhausted but the input source might produce more data in
 * the future. This new data would be represented by new input handles.
 * {@link #NO_DATA_PERMANENT} means that the input source is permanently
 * exhausted. There would be no more input handles for this input source.
 * <p>
 * <i>Thread-safety:</i> Implementations are not required to be thread-safe.
 * </p>
 */
public interface BinaryInput {

    /**
     * Data source is depleted temporarily. More data might be available in the future.
     */
    public static int NO_DATA_TEMPORARY = -2;

    /**
     * Data source is depleted permanently. No more data will ever be received
     * from this source in the future.
     */
    public static int NO_DATA_PERMANENT = -1;

    /**
     * Reads a byte from the input source.
     *
     * @return data byte (a non-negative number) or {@link #NO_DATA_TEMPORARY}
     *         if the source is temporarily depleted or
     *         {@link #NO_DATA_PERMANENT} if the source is permanently depleted
     * @throws IOException I/O error
     */
    int read() throws IOException;

    /**
     * Reads a sequence of bytes from the input source.
     *
     * @param b buffer where to store the bytes
     * @param off starting location in the buffer
     * @param len maximum number of bytes to be read
     * @return the number of bytes (a non-negative number, could be smaller than
     *         <code>len</code>, even zero) or {@link #NO_DATA_TEMPORARY} if the
     *         source is temporarily depleted or {@link #NO_DATA_PERMANENT} if
     *         the source is permanently depleted
     * @throws IOException I/O error
     * @throws IndexOutOfBoundsException if the combination of
     *             <code>b.length</code>, <code>off</code> and <code>len</code>
     *             is not valid
     * @throws NullPointerException if <code>b</code> is <code>null</code>
     */
    int read(byte[] b, int off, int len) throws IOException;
}
