/* **********************************************************
 * Copyright (c) 2016, 2019-2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.protocol.common.json;

import java.io.IOException;
import java.io.OutputStream;
import java.text.MessageFormat;
import java.util.Map.Entry;

import org.apache.commons.codec.binary.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerationException;
import com.fasterxml.jackson.core.JsonGenerator;
import com.vmware.vapi.CoreException;
import com.vmware.vapi.data.BlobValue;
import com.vmware.vapi.data.BooleanValue;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.DoubleValue;
import com.vmware.vapi.data.IntegerValue;
import com.vmware.vapi.data.ListValue;
import com.vmware.vapi.data.OptionalValue;
import com.vmware.vapi.data.SecretValue;
import com.vmware.vapi.data.StringValue;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.internal.protocol.common.DirectSerializer;
import com.vmware.vapi.internal.protocol.common.SerializerConfig;
import com.vmware.vapi.internal.util.Validate;

/**
 * {@code DirectSerializer} implementation which outputs JSON.
 */
public class JsonDirectSerializer implements DirectSerializer {
    private static final Logger LOGGER = LoggerFactory.getLogger(JsonDirectSerializer.class);

    private JsonFactory factory;

    public JsonDirectSerializer(JsonFactory factory) {
        Validate.notNull(factory);
        this.factory = factory;
    }

    @Override
    public void serialize(DataValue value,
                          OutputStream outputStream,
                          SerializerConfig serializerConfig) throws IOException {
        JsonGenerator generator = null;
        try {
            // TODO: rest native: encoding; this defaults to UTF-8;
            //       just accept a Writer instead? or allow specification of encoding
            //
            // rfc7159: JSON text SHALL be encoded in UTF-8, UTF-16, or UTF-32.
            //          The default encoding is UTF-8.
            generator = factory.createGenerator(outputStream);
            if (serializerConfig.isPrettyPrint()) {
                generator.useDefaultPrettyPrinter();
            }

            writeValue(generator, value, null);
            generator.flush();
        } finally {
            if (serializerConfig.isCloseRequired() && generator != null) {
                generator.close();
            }
        }
    }

    private void writeValue(JsonGenerator generator,
                            DataValue dataValue,
                            StructValue parentStructValue)
            throws JsonGenerationException, IOException {
        switch (dataValue.getType()) {
            case STRING:
                generator.writeString(((StringValue) dataValue).getValue());
                break;
            case INTEGER:
                generator.writeNumber(((IntegerValue) dataValue).getValue());
                break;
            case DOUBLE:
                writeDoubleValue((DoubleValue) dataValue, generator);
                break;
            case BOOLEAN:
                generator.writeBoolean(((BooleanValue) dataValue).getValue());
                break;
            case BLOB:
                generator.writeString(Base64.encodeBase64String(((BlobValue) dataValue).getValue()));
                break;
            case VOID:
                // VOID can only be top level
                break;
            case SECRET:
                // Char array - we write as String
                generator.writeString(new String(((SecretValue) dataValue).getValue()));
                break;
            case OPTIONAL:
                DataValue nestedValue = ((OptionalValue) dataValue).getValue();
                if (nestedValue != null) {
                    writeValue(generator, nestedValue, parentStructValue);
                } else {
                    generator.writeNull();
                }
                break;
            case LIST:
                ListValue listValue = ((ListValue) dataValue);
                generator.writeStartArray();
                for (DataValue nestedData : listValue) {
                    writeValue(generator, nestedData, parentStructValue);
                }
                generator.writeEndArray();
                break;
            case STRUCTURE:
            case ERROR:
                StructValue structValue = ((StructValue) dataValue);
                generator.writeStartObject();
                for (Entry<String, DataValue> entry : structValue.getFields().entrySet()) {
                    generator.writeFieldName(entry.getKey());
                    writeValue(generator, entry.getValue(), structValue);
                }
                generator.writeEndObject();
                break;
            default:
                LOGGER.error("Unrecognized DataType {0}", dataValue.getType());

                throw new CoreException(MessageFormat.format("Unrecognized DataType {0}",
                        dataValue.getType()));
        }
    }

    private void writeDoubleValue(DoubleValue doubleVal, JsonGenerator generator)
            throws IOException {
        double d = doubleVal.getValue();
        if (Double.isNaN(d) || Double.isInfinite(d)) {
            throw new CoreException("JSON doesn't support the double value: " + d);
        }
        generator.writeNumber(d);
    }
}
