/* **********************************************************
 * Copyright (c) 2012, 2017, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.protocol.server.rpc;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collections;
import java.util.Map;

import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.internal.util.io.IoUtil;
import com.vmware.vapi.protocol.server.rpc.RequestReceiver;
import com.vmware.vapi.protocol.server.rpc.RequestReceiver.RequestContext;


/**
 * Transport context which wraps an output stream and ignores every response
 * except the final one. It is suitable for transports protocols which do not
 * support multiple response frames for one request.
 */
public final class OutputStreamTransportContext implements
        RequestReceiver.TransportContext {

    private final OutputStream out;

    /**
     *
     * @param out output stream used to write responses; must not be
     *            <code>null</code>
     */
    public OutputStreamTransportContext(OutputStream out) {
        Validate.notNull(out);
        this.out = out;
    }

    @Override
    public void send(InputStream response, int responseLength, boolean isFinal)
            throws IOException {
        Validate.notNull(response);
        if (isFinal) {
            IoUtil.copy(response, out);
            out.flush();
            out.close();
        }// else - ignore the non-final messages, because the transport does
        //  not support multiple response frames for one request
    }

    @Override
    public RequestContext getRequestContext() {
        return new RequestContext() {
            @Override
            public String getUserAgent() {
                return null;
            }

            @Override
            public String getSession() {
               return null;
            }

            @Override
            public String getServiceId() {
               return null;
            }

            @Override
            public String getOperationId() {
               return null;
            }

            @Override
            public String getAcceptLanguage() {
                return null;
            }

            @Override
            public Map<String, String> getAllProperties() {
               return Collections.emptyMap();
            }

        };
    }

    @Override
    public void setHeader(String name, String value) {
       throw new UnsupportedOperationException();
    }
}
