/* **********************************************************
 * Copyright (c) 2012-2015,2018 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.util;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.ExecutionContext.SecurityContext;
import com.vmware.vapi.security.AuthenticationHandler.AuthenticationResult;
import com.vmware.vapi.security.PrincipalId;

/**
 * Utility methods related to authentication.
 */
public class AuthenticationUtil {
   private static final Logger logger =
        LoggerFactory.getLogger(AuthenticationUtil.class);

   /**
    * Retrieve authentication result from the given execution context.
    *
    * @param ctx execution context to retrieve authentication result from
    * @return authentication result; maybe {@code null} if
    *         {@link SecurityContext} is not set in
    *         <code>ctx</code>.
    */
   public AuthenticationResult retrieveAuthnResult(ExecutionContext ctx) {
        SecurityContext secCtx = ctx.retrieveSecurityContext();
        if (secCtx == null) {
            logger.debug("No SecurityContext found");
            return null;
        }

        return (AuthenticationResult) secCtx.getProperty(
            SecurityContext.AUTHENTICATION_DATA_ID);
    }

    /**
     * Retrieve user name from the given authentication result.
     *
     * @param authnResult authentication result to retrieve user name from
     * @return user name; maybe {@code null} if there is no user in
     *         <code>authnResult</code>.
     */
    public String retrieveUserName(AuthenticationResult authnResult) {
        PrincipalId userId = authnResult.getUser();
        if (userId == null) {
            return null;
        }
        if (userId.getDomain() != null) {
            logger.debug("User name {}, domain {} ",
                userId.getName(), userId.getDomain());
            return userId.getDomain() + "\\" + userId.getName();
        } else {
            logger.debug("User name {}", userId.getName());
            return userId.getName();
        }
    }

    /**
     * Retrieve user groups from the given authentication result.
     *
     * @param authnResult authentication result to retrieve user groups from
     * @return group names; maybe an empty list if there is no group
     *         in <code>authnResult</code>.
     */
    public Set<String> retrieveUserGroups(AuthenticationResult authnResult) {
        List<PrincipalId> groupIds = authnResult.getGroups();
        Set<String> groupNames = new HashSet<String>();
        if (groupIds != null) {
            for (PrincipalId groupId : groupIds) {
                String groupName;
                if (groupId.getDomain() != null) {
                    groupName = groupId.getDomain() + "\\" + groupId.getName();
                } else {
                    groupName = groupId.getName();
                }
                groupNames.add(groupName);
            }
        }
        logger.debug("Group names {}", groupNames);
        return groupNames;
    }
}
