/* *************************************************************************
 * Copyright (c) 2019, 2021-2022 VMware, Inc. All rights reserved. -- VMware Confidential
 * *************************************************************************/

package com.vmware.vapi.internal.util;

import java.util.Collection;
import java.util.Iterator;
import java.util.zip.CRC32;

/**
 * This class is for internal use.
 */
public class StringUtils {

    public static final String EMPTY = "";
    public static final int INDEX_NOT_FOUND = -1;
    private static final String MASK = "********";
    private static final int MASK_LEN = MASK.length();

    /**
     * Strips a specified set of characters from the start and end of an input
     * string.
     *
     * <p>A {@code null} input String returns {@code null}. An empty string ("")
     * input returns the empty string.</p>
     *
     * @param input the String to remove characters from, may be {@code null}
     * @param toStrip the characters to remove, can not be {@code null}.
     * @return the stripped String
     */
    public static String strip(String input, String toStrip) {
        int strLen;
        if (input == null || toStrip == null || (strLen = input.length()) == 0) {
            return input;
        }
        int start = 0;
        while (start != strLen
               && toStrip.indexOf(input.charAt(start)) != INDEX_NOT_FOUND) {
            start++;
        }
        int end = input.length();
        if (start == strLen) {
            return EMPTY;
        }
        while (end != 0 && toStrip.indexOf(input.charAt(end - 1)) != INDEX_NOT_FOUND) {
            end--;
        }
        return input.substring(start, end);
    }

    /**
     * Strips occurrences of a set of characters from the end of a String.
     *
     * <p>A {@code null} input String returns {@code null}.</p>
     * <p>An empty string ("") input returns the empty string.</p>
     *
     * @param input the String to remove characters from, may be {@code null}
     * @param toStrip the characters to remove, can not be {@code null}.
     * @return the stripped String
     */
    public static String stripEnd(String input, String strip) {
        int end;
        if (input == null || (end = input.length()) == 0) {
            return input;
        }

        while (end != 0 && strip
                .indexOf(input.charAt(end - 1)) != INDEX_NOT_FOUND) {
            end--;
        }
        return input.substring(0, end);
    }

    /**
     * Checks if a String is {@code null}, empty or consists of whitespaces only.
     *
     * @param input the String to check, may be {@code null}
     * @return {@code true} if the String is {@code null}, empty or whitespace
     */
    public static boolean isBlank(String input) {
        if (isEmpty(input) || (input = input.trim()).isEmpty()) {
            return true;
        }
        return false;
    }

    /**
     * Checks if a String is not whitespace, empty ("") or {@code null}.
     *
     * @param input the String to check, may be {@code null}
     * @return {@code false} if the String is {@code null}, empty or whitespace
     */
    public static boolean isNotBlank(String input) {
        return !isBlank(input);
    }

    /**
     * Checks if a String is empty ("") or {@code null}.
     *
     * @param input the String to check, may be {@code null}.
     * @return {@code true} if the string is {@code null}, empty or whitespace
     */
    public static boolean isEmpty(String input) {
        return input == null || input.length() == 0;
    }

    /**
     * Uncapitalizes an input string.
     *
     * @param input the String to uncapitalize, may be {@code null}.
     * @return the uncapitalized string
     */
    public static String uncapitalize(String input) {
        if (input == null || input.length() == 0) {
            return input;
        }
        char[] c = input.toCharArray();
        c[0] = Character.toLowerCase(c[0]);
        input = new String(c);
        return input;
    }

    /**
     * Capitalizes an input string.
     *
     * @param input the String to uncapitalize, may be {@code null}.
     * @return the uncapitalized string
     */
    public static String capitalize(String input) {
        if (input == null || input.length() == 0) {
            return input;
        }
        char c[] = input.toCharArray();
        c[0] = Character.toUpperCase(c[0]);
        input = new String(c);
        return input;
    }

    /**
     * Retrieves a part of the input string after the given delimiter.
     *
     * @param input the input string, may not be {@code null}.
     * @para delimiter the delimiter to separate the string to retrieve, may not
     *       be {@code null}
     * @return the substring after the provided delimiter.
     */
    public static String substringAfter(String input, String delimiter) {
        int pos = input.indexOf(delimiter);
        if (pos == INDEX_NOT_FOUND) {
            return StringUtils.EMPTY;
        }
        return input.substring(pos + delimiter.length());
    }

    /**
     * Compares two Strings, returning {@code true} if they are equal.
     *
     * <p>{@code null} is handled without exceptions. Two {@code null}
     * references are considered equal. The comparison is case-sensitive.</p>
     *
     * @param str1  the first String, may be {@code null}
     * @param str2  the second String, may be {@code null}
     * @return <code>true</code> if the Strings are equal, case sensitive, or
     *  both {@code null}
     */
    public static boolean equals(String str1, String str2) {
        return str1 == null ? str2 == null : str1.equals(str2);
    }

    /**
     * Count how many times a substring appears in the larger string.
     * A {@code null} or empty ("") String input returns {@code 0}.
     *
     * @param str the String to check, may be {@code null}
     * @param sub the substring to count, may be {@code null}
     * @return the number of occurrences, 0 if either String is {@code null}
     */
    public static int countMatches(String str, String sub) {
        if (StringUtils.isEmpty(str)|| StringUtils.isEmpty(sub)) {
            return 0;
        }
        int count = 0;
        int idx = 0;
        while ((idx = str.indexOf(sub, idx)) != -1) {
            count++;
            idx += sub.length();
        }
        return count;
    }

    /**
     * Get the ordinal index of a given string.
     *
     * @param input the input string to search in.
     * @param searchStr the string to search for.
     * @param ordinal the ordinal number.
     * @return the index of the string.
     */
    public static int ordinalIndexOf(String input,
                                     String searchStr,
                                     int ordinal) {
        if (input == null || searchStr == null || searchStr.isEmpty() || ordinal <= 0) {
            return INDEX_NOT_FOUND;
        }
        int found = 0;
        int index = INDEX_NOT_FOUND;
        do {
            index = input.indexOf(searchStr, ++index);

            if (index < 0) {
                return index;
            }
            found++;
        } while (found < ordinal);
        return index;
    }

    /**
     * Checks if a String is numeric, i.e. whether it contains only digit
     * characters.
     * <p> The method takes into account the following digit types:
     * <ul>
     * <li> ISO-Latin-1 </li>
     * <li> Arabic-Indic </li>
     * <li> Extended Arabic-Indic</li>
     * <li> Devanagari </li>
     * <li> Fullwidth </li>
     *
     * <p> The method will return {@code true} for an empty input string and
     * {@code false} for {@code null}. </p>
     *
     * @param input the String to check, may be {@code null}
     * @return {@code true} if only contains digits, and is non-null
     */
    public static boolean isNumeric(String input) {
        if (input == null) {
            return false;
        }
        for (int i = 0; i < input.length(); i++) {
            if (!Character.isDigit(input.charAt(i))) {
                return false;
            }
        }
        return true;
    }

    /**
     * Removes a substring only if it is at the end of a source string,
     * otherwise returns the source string.
     */
    public static String removeEnd(String input, String end) {
        if (input.endsWith(end)) {
            return input.substring(0, input.length() - end.length());
        }
        return input;
    }

    /**
     * Joins the elements of the provided {@code Collection} into a single
     * String containing the provided elements. Elements of value {@code null}
     * are treated as empty String values ("").
     *
     * <p>No delimiter is added before or after the list. A {@code null} separator
     * is the same as an empty String ("").</p>
     *
     * @param collection the {@code Collection} of values to join together, may
     *        be {@code null}
     * @param separator the separator character to use, {@code null} is treated as ""
     * @return the joined String, {@code null} when the provided input is {@code null}.
     */
    public static String join(Collection<?> collection, String separator) {
        if (collection == null) {
            return null;
        }
        if (collection.isEmpty()) {
            return EMPTY;
        }
        if (separator == null) {
            separator = EMPTY;
        }
        Iterator<?> iterator = collection.iterator();
        Object first = iterator.next();
        if (!iterator.hasNext()) {
            return first == null ? EMPTY : first.toString();
        }

        // two or more elements
        StringBuilder sb = new StringBuilder();
        if (first != null) {
            sb.append(first);
        }

        while (iterator.hasNext()) {
            Object obj = iterator.next();
            sb.append(separator);
            if (obj == null) {
                continue;
            }
            sb.append(obj);
        }
        return sb.toString();
    }

    /**
     * Joins the elements of the provided array into a single String containing
     * the provided array of elements. Elements of value {@code null} are treated
     * as empty String values ("").
     *
     * @param array the array of values to join together, may
     *        be {@code null}
     * @param separator the separator character to use, {@code null} is treated as ""
     * @return the joined String, {@code null} if null iterator input.
     */
    public static String join(Object[] array, String separator) {
        if (array == null) {
            return null;
        }
        if (separator == null) {
            separator = EMPTY;
        }
        StringBuilder buf = new StringBuilder();

        for (int i = 0; i < array.length; i++) {
            if (i != 0) {
                buf.append(separator);
            }
            if (array[i] != null) {
                buf.append(array[i]);
            }
        }
        return buf.toString();
    }

    /**
     * Returns the {@link CRC32} checksum value as hexadecimal string
     *
     * @param checksum
     * @return A string representation of the CRC-32 checksum value in base-16.
     */
    public static String crc32ToHexString(CRC32 checksum) {
        return String.format("%08x", checksum.getValue());
    }

    /**
     * Replaces the last 8 characters of a string with the asteriks symbol. If
     * the string is shorter than 8 symbols, replaces all of its characters.
     *
     * @param sensitiveString the string to mask
     * @return A string with the last 8 characters replaced with "*"
     */
    public static String mask(String sensitiveString) {
        if (sensitiveString == null) {
            return null;
        }
        if (sensitiveString.length() < MASK_LEN) {
            return MASK.substring(0, sensitiveString.length());
        }
        int trimLen = sensitiveString.length() - MASK_LEN;
        String masked = sensitiveString.substring(0, trimLen) + MASK;
        return masked;
    }
}
