/* *************************************************************************
 * Copyright (c) 2019 VMware, Inc. All rights reserved. -- VMware Confidential
 * *************************************************************************/

package com.vmware.vapi.internal.util;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;

/**
 * This class is for internal use.
 */
public final class Validate {

    /**
     * Validate that the specified method input is not {@code null}.
     *
     * <p>Default message of the exception is &quot;The validated object is null&quot;.</p>
     *
     * @param object the input object
     * @throws IllegalArgumentException if the object is {@code null}
     */
    public static void notNull(Object object) {
        notNull(object, "The validated object is null");
    }

    /**
     * Validate that the specified argument is not {@code null}; else throw
     * an exception with the specified message.
     *
     * @param object the input object
     * @param message the exception message if the input object is {@code null}
     */
    public static void notNull(Object object, String message) {
        if (object == null) {
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Validate that the argument condition is <code>true</code>; else
     * throw an exception.
     *
     * @param expression the boolean expression to check
     * @throws IllegalArgumentException if expression is {@code false}
     */
    public static void isTrue(boolean expression) {
        if (!expression) {
            throw new IllegalArgumentException("The validated expression is false");
        }
    }

    /**
     * Validate that the argument condition is {@code true}.
     *
     * @param expression the boolean expression to check
     * @param message the exception message if invalid
     * @throws IllegalArgumentException if expression is <code>false</code>
     */
    public static void isTrue(boolean expression, String message) {
        if (expression == false) {
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Validate that the specified argument collection is neither {@code null}
     * nor of a size of zero (no elements).
     *
     * @param collection the collection to check
     * @throws IllegalArgumentException if the collection is empty
     */
    public static void notEmpty(Collection<?> collection) {
        if (collection == null || collection.size() == 0) {
            throw new IllegalArgumentException("The validated collection is empty");
        }
    }

    /**
     * Validate that the specified argument array is neither {@code null}
     * nor with a length of zero (no elements).
     *
     * @param array the array to check
     * @throws IllegalArgumentException if the array is empty
     */
    public static void notEmpty(Object[] array) {
        if (array == null || array.length == 0) {
            throw new IllegalArgumentException("The validated array is empty");
        }
    }

    /**
     * Validate that the specified argument string is
     * neither {@code null} nor a length of zero (no characters).
     *
     * @param string the string to check
     * @throws IllegalArgumentException if the string is empty
     */
    public static void notEmpty(String string) {
        if (string == null || string.length() == 0) {
            throw new IllegalArgumentException("The validated string is empty");
        }
    }

    /**
     * Validate that the specified argument map is neither {@code null}
     * nor a size of zero (no elements).
     *
     * <p>The message in the exception is &quot;The validated map is
     * empty&quot;.</p>
     *
     * @param map the map to check
     * @throws IllegalArgumentException if the map is empty
     */
    public static void notEmpty(Map<?, ?> map) {
        notEmpty(map, "The validated map is empty");
    }

    /**
     * Validate that the specified argument map is neither {@code null}
     * nor a size of zero (no elements).
     *
     * @param map the map to check
     * @param message the exception message if invalid
     * @throws IllegalArgumentException if the map is empty
     */
    public static void notEmpty(Map<?, ?> map, String message) {
        if (map == null || map.size() == 0) {
            throw new IllegalArgumentException(message);
        }
    }

    /**
     * Validate that the specified argument array is neither
     * {@code null} nor contains any elements that are {@code null}.
     *
     * @param array the array to check
     * @throws IllegalArgumentException if the array is {@code null} or
     * an element in the array is {@code null}
     */
    public static void noNullElements(Object[] array) {
        Validate.notNull(array);
        for (int i = 0; i < array.length; i++) {
            if (array[i] == null) {
                throw new IllegalArgumentException("The validated array contains null element at index: " + i);
            }
        }
    }

    /**
     * Validate that the specified argument collection is neither
     * {@code null} nor contains any elements that are {@code null}.
     *
     * @param collection  the collection to check
     * @param message  the exception message if the collection has
     * @throws IllegalArgumentException if the collection is {@code null} or
     * an element in the collection is {@code null}
     */
    public static void noNullElements(Collection<?> collection, String message) {
        Validate.notNull(collection);
        for (Iterator<?> it = collection.iterator(); it.hasNext();) {
            if (it.next() == null) {
                throw new IllegalArgumentException(message);
            }
        }
    }

    /**
     * Validate that the specified argument collection is neither
     * {@code null} nor contains any elements that are {@code null};
     * else throw an exception with the specified message.
     *
     * @param collection the collection to check
     * @throws IllegalArgumentException if the collection is {@code null} or
     * an element in the collection is {@code null}
     */
    public static void noNullElements(Collection<?> collection) {
        Validate.notNull(collection);
        int i = 0;
        for (Iterator<?> it = collection.iterator(); it.hasNext(); i++) {
            if (it.next() == null) {
                throw new IllegalArgumentException("The validated collection contains null element at index: "
                                                   + i);
            }
        }
    }

}
