/* **********************************************************
 * Copyright 2013, 2019 VMware, Inc. All rights reserved.
 *      -- VMware Confidential
 * *********************************************************
 */

/*
 * DefaultStringLocalizer.java --
 *
 *      Utility for localization of LocalizableMessage
 */
package com.vmware.vapi.l10n;

import java.util.List;
import java.util.Locale;
import java.util.ResourceBundle;

import com.vmware.vapi.CoreException;
import com.vmware.vapi.data.ListValue;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.std.LocalizableMessage;
import com.vmware.vapi.std.StandardDataFactory;

/**
 * Utility for localization of {@link LocalizableMessage} or its dynamic
 * representation as {@link StructValue}.
 *
 * <p>Uses {@link ResourceBundleProvider} to locate resource bundle for
 * message localization (i.e. resolving the message key}.
 *
 * <p>Uses {@link TemplateFormatter} to format the message template.
 */
public final class MessageLocalizer {
    private TemplateFormatter formatter;
    private ResourceBundleProvider bundleProvider;
    private Locale locale;

    /**
     * Constructor.
     *
     * @param formatter  formatter for message templates
     * @param bundleProvider  provider for message bundles
     * @param locale  target locale;
     * @throws IllegalArgumentException if any of the parameters
     *         is <code>null</code>
     */
    public MessageLocalizer(TemplateFormatter formatter,
                            ResourceBundleProvider bundleProvider,
                            Locale locale) {
        Validate.notNull(formatter);
        Validate.notNull(bundleProvider);
        Validate.notNull(locale);

        this.formatter = formatter;
        this.bundleProvider = bundleProvider;
        this.locale = locale;
    }

    /**
     * Localizes a <code>LocalizableMessage</code> for given target locale.
     *
     * @param localizable message to be localized
     * @param locale target locale
     * @return localized message for the target locale
     * @throws IllegalArgumentException if any of the parameters
     *         is <code>null</code>
     */
    public String localize(LocalizableMessage localizable, Locale locale) {
        Validate.notNull(localizable);
        Validate.notNull(locale);

        return doLocalize(localizable.getId(), localizable.getArgs(), locale);
    }

    /**
     * Localizes a <code>LocalizableMessage</code> for the locale used to
     * create <code>this</code> instance.
     *
     * @see #localize(LocalizableMessage, Locale)
     */
    public String localize(LocalizableMessage localizable) {
        return localize(localizable, this.locale);
    }

    /**
     * Localizes a dynamic (<code>StructValue</code>) representation of
     * localizable message.
     *
     * @param localizable localizable message
     * @param locale target locale
     * @return localized message for the target locale
     * @throws LocalizationException if the specified <code>localizable</code>
     *         doesn't have the expected structure for localizable message
     * @throws IllegalArgumentException if any of the parameters
     *         is <code>null</code>
     */
    public String localize(StructValue localizable, Locale locale) {
        Validate.notNull(localizable);
        Validate.notNull(locale);

        try {
            String msgId = localizable.getString(
                    StandardDataFactory.ID_FIELD_NAME);

            ListValue argsListValue = localizable.getList(
                    StandardDataFactory.ARGS_FIELD_NAME);

            return doLocalize(msgId, argsListValue.getStringList(), locale);
        } catch (CoreException ex) {
            throw new LocalizationException(
                "Specified StructValue is not representing localizable message");
        }
    }

    /**
     * Localizes a dynamic (<code>StructValue</code>) representation of
     * localizable message for the locale used to create <code>this</code>
     * instance.
     *
     * @see #localize(LocalizableMessage, Locale)
     */
    public String localize(StructValue localizable) {
        return localize(localizable, this.locale);
    }

    String doLocalize(String msgId, List<String> args, Locale locale) {
        Validate.notNull(msgId);
        Validate.notNull(args);

        ResourceBundle msgCatalog =
            bundleProvider.getResourceBundle(msgId, locale);
        String msgTemplate = msgCatalog.getString(msgId);

        return formatter.format(msgTemplate, args, locale);
    }
}
