/* **********************************************************
 * Copyright 2013, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/

/*
 * StringFormatTemplateFormatter.java --
 *
 *      java.util.Formatter based TemplateFormatter
 */
package com.vmware.vapi.l10n;

import java.util.Calendar;
import java.util.Formatter;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.TimeZone;
import java.util.UnknownFormatConversionException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.vmware.vapi.internal.util.Validate;

/**
 * {@link Formatter} based <code>TemplateFormatter</code>.
 */
public final class StringFormatTemplateFormatter implements TemplateFormatter {

    // %[argument_index$][flags][width][.precision][t]conversion
    private static final String formatSpecifier =
        "%(\\d+\\$)?([-#+ 0,(\\<]*)?(\\d+)?(\\.\\d+)?([tT])?([a-zA-Z%])";

    private static final Pattern fsPattern = Pattern.compile(formatSpecifier);

    private final MessageArgConverter argConverter = new MessageArgConverter();

    @Override
    public String format(String msgTemplate, List<String> args, Locale locale) {
        return format(msgTemplate, args, locale, null);
    }

    @Override
    public String format(String msgTemplate,
                         List<String> args,
                         Locale locale,
                         TimeZone tz) {
        Validate.notNull(msgTemplate);
        Validate.notNull(args);
        Validate.notNull(locale);

        try(Formatter formatter = new Formatter(locale)) {
            formatter.format(msgTemplate,
                             buildTypedArguments(args, msgTemplate, tz));

            return formatter.toString();
        } catch (RuntimeException ex) {
            throw new LocalizationException("Unable to format message template",
                                            ex);
        }
    }

    Object[] buildTypedArguments(List<String> argValues,
                                 String msgTemplate,
                                 TimeZone tz) {
        Object[] result = new Object[argValues.size()];
        List<FormatSpecifier> formats = parse(msgTemplate);

        int implicitArgIndex = 0;
        int argIndex = 0;
        for (FormatSpecifier fStr : formats) {
            if (fStr.getIndex() < 0) {
                continue;
            }

            if (fStr.getIndex() > 0) {
                argIndex = fStr.getIndex() - 1;
            } else {   // fStr.index == 0
                // 0 means no explicit position (like %3$) specified,
                // i.e. it is just %s; this needs separate _independent_
                // (of the other explicit position specifiers) counter
                argIndex = implicitArgIndex++;
            }

            if (result[argIndex] == null) {   // not processed yet
                result[argIndex] = buildTypedArg(fStr,
                                                 argValues.get(argIndex),
                                                 tz);
            }
        }

        return result;
    }

    Object buildTypedArg(FormatSpecifier fStr, String argValue, TimeZone tz) {
        if (Conversion.isInteger(fStr.getConversion())) {
            return argConverter.toLong(argValue);
        }
        if (Conversion.isFloat(fStr.getConversion())) {
            return argConverter.toDouble(argValue);
        }
        if (fStr.isDateTime) {
            Calendar dt = argConverter.toCalendarArgument(argValue);
            if (tz != null) {
                dt.setTimeZone(tz);
            }
            return dt;
        }

        // do not convert
        return argValue;
    }

    // Look for format specifiers in the format string.
    List<FormatSpecifier> parse(String s) {
        List<FormatSpecifier> al = new LinkedList<FormatSpecifier>();
        Matcher m = fsPattern.matcher(s);
        int i = 0;
        while (i < s.length()) {
            if (m.find(i)) {
                // there are 6 groups in the regular expression
                String[] tokens = new String[6];
                for (int j = 0; j < m.groupCount(); j++) {
                    tokens[j] = m.group(j + 1);
                }
                al.add(new FormatSpecifier(tokens));
                i = m.end();
            } else {
                break;
            }
        }

        return al;
    }

    class FormatSpecifier {
        private int index = -1;
        boolean isDateTime = false;
        private char c;

        FormatSpecifier(String[] tokens) {
            parseIndex(tokens[0]);       // index
            if (tokens[4] != null) {     // date - 'T' or 't'
                isDateTime = true;
            }
            parseConversion(tokens[5]);  // conversion char
        }

        private void parseIndex(String s) {
            if (s != null) {
                // regexp only matches digits; should always succeed
                index = Integer.parseInt(s.substring(0, s.length() - 1));
            } else {
                // no index specified
                index = 0;
            }
        }

        public int getIndex() {
            return index;
        }

        private char parseConversion(String s) {
            c = s.charAt(0);
            if (!isDateTime) {
                if (!Conversion.isValid(c))
                    throw new UnknownFormatConversionException(String.valueOf(c));
                // TODO: conversion could be upper-case or lower-case char - ensure can handle both
                c = Character.toLowerCase(c);
                if (Conversion.isText(c))
                    // no argument value for this specifier
                    index = -2;
            }
            return c;
        }

        private char getConversion() {
            return c;
        }
    }

    private static class Conversion {
        // Byte, Short, Integer, Long, BigInteger
        // (and associated primitives due to autoboxing)
        static final char DECIMAL_INTEGER     = 'd';
        static final char OCTAL_INTEGER       = 'o';
        static final char HEXADECIMAL_INTEGER = 'x';
        static final char HEXADECIMAL_INTEGER_UPPER = 'X';

        // Float, Double, BigDecimal
        // (and associated primitives due to autoboxing)
        static final char SCIENTIFIC          = 'e';
        static final char SCIENTIFIC_UPPER    = 'E';
        static final char GENERAL             = 'g';
        static final char GENERAL_UPPER       = 'G';
        static final char DECIMAL_FLOAT       = 'f';
        static final char HEXADECIMAL_FLOAT   = 'a';
        static final char HEXADECIMAL_FLOAT_UPPER = 'A';

        // Character, Byte, Short, Integer
        // (and associated primitives due to autoboxing)
        static final char CHARACTER           = 'c';
        static final char CHARACTER_UPPER     = 'C';

        // Boolean
        static final char BOOLEAN             = 'b';
        static final char BOOLEAN_UPPER       = 'B';

        // String
        static final char STRING              = 's';
        static final char STRING_UPPER        = 'S';

        static final char LINE_SEPARATOR      = 'n';
        static final char PERCENT_SIGN        = '%';

        static boolean isValid(char c) {
            return (isGeneral(c) || isInteger(c) || isFloat(c) || isText(c)
                    || c == 't' || isCharacter(c));
        }

        // Returns true iff the Conversion is applicable to all objects.
        static boolean isGeneral(char c) {
            switch (c) {
            case BOOLEAN:
            case BOOLEAN_UPPER:
            case STRING:
            case STRING_UPPER:
                return true;
            default:
                return false;
            }
        }

        // Returns true iff the Conversion is applicable to character.
        static boolean isCharacter(char c) {
            switch (c) {
            case CHARACTER:
            case CHARACTER_UPPER:
                return true;
            default:
                return false;
            }
        }

        // Returns true iff the Conversion is an integer type.
        static boolean isInteger(char c) {
            switch (c) {
            case DECIMAL_INTEGER:
            case OCTAL_INTEGER:
            case HEXADECIMAL_INTEGER:
            case HEXADECIMAL_INTEGER_UPPER:
                return true;
            default:
                return false;
            }
        }

        // Returns true iff the Conversion is a floating-point type.
        static boolean isFloat(char c) {
            switch (c) {
            case SCIENTIFIC:
            case SCIENTIFIC_UPPER:
            case GENERAL:
            case GENERAL_UPPER:
            case DECIMAL_FLOAT:
            case HEXADECIMAL_FLOAT:
            case HEXADECIMAL_FLOAT_UPPER:
                return true;
            default:
                return false;
            }
        }

        // Returns true iff the Conversion does not require an argument
        static boolean isText(char c) {
            switch (c) {
            case LINE_SEPARATOR:
            case PERCENT_SIGN:
                return true;
            default:
                return false;
            }
        }
    }

}
