/* **********************************************************
 * Copyright (c) 2012-2013, 2020-2022 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.protocol;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.Executor;

import com.vmware.vapi.core.ApiProvider;
import com.vmware.vapi.core.MethodResult;
import com.vmware.vapi.tracing.Tracer;

/**
 * This class is used for vAPI runtime client configuration. The class is
 * immutable.
 */
public final class ClientConfiguration {

    private final List<RequestProcessor> proc;
    private final Executor executor;
    private final boolean hasUserDefinedProc;
    boolean steamingEnabled;
    private final Tracer tracer;

    private ClientConfiguration(List<RequestProcessor> requestProcessors,
                                Executor executor,
                                boolean steamingEnabled,
                                Tracer tracer) {
        this.hasUserDefinedProc = requestProcessors != null;
        this.proc = Collections
                .unmodifiableList(createRequestProcessorsList(requestProcessors));
        this.executor = executor;
        this.steamingEnabled = steamingEnabled;
        this.tracer = Objects.requireNonNull(tracer);
    }

    /**
     * @return the request processor chain. cannot be null.
     */
    public List<RequestProcessor> getRequestProcessors() {
        return proc;
    }

    /**
     * Returns the executor which must be used by the client to process
     * requests. If there is an executor configured for the client, some
     * processing will happen in the executor, instead of the caller thread. The
     * methods executed asynchronously will depend on the client and its
     * configuration (i.e. sync / async stub, BIO/NIO...).
     *
     * @return the executor the client must use to process requests.
     */
    public Executor getExecutor() {
        return executor;
    }

    /**
     * Configures the client stack to advertise or not streaming capability.
     *
     * @return if {@code true} streaming capability will be advertised in the
     *         {@code accept} HTTP header.
     */
    public boolean isSteamingEnabled() {
        return steamingEnabled;
    }

    /**
     * Returns the {@link Tracer} configured for the vAPI runtime.
     *
     * @return the tracer. Cannot be {@code null}
     */
    public Tracer getTracer() {
        return tracer;
    }

    /**
     * Creates the request processor chain.
     *
     * @return the request processor chain. cannot be null.
     */
    private ArrayList<RequestProcessor> createRequestProcessorsList(List<RequestProcessor> additionalProcessors) {
        ArrayList<RequestProcessor> proc = new ArrayList<RequestProcessor>();
        if (additionalProcessors != null) {
            proc.addAll(additionalProcessors);
        }

        return proc;
    }

    /**
     * @return true if the user has defined own processor list
     */
    private boolean hasUserDefinedProc() {
        return hasUserDefinedProc;
    }

    /**
     * This is a builder for {@link ClientConfiguration}
     */
    public static final class Builder {
        private List<RequestProcessor> requestProcessors;
        private Executor executor;
        private boolean streamingEnabled = true;
        private Tracer tracer = Tracer.NO_OP;

        public Builder() {
        }

        /**
         * Create a builder that is preconfigured with the given
         * {@link ClientConfiguration}
         *
         * @param config can be <code>null</code>
         */
        public Builder(ClientConfiguration config) {
            if (config != null) {
                this.executor = config.getExecutor();
                if (config.hasUserDefinedProc()) {
                    this.requestProcessors = new ArrayList<RequestProcessor>(config
                            .getRequestProcessors());
                }
                this.streamingEnabled = config.isSteamingEnabled();
                if (config.getTracer() != null) {
                    this.tracer = config.getTracer();
                }
            }
        }

        /**
         * @param processors the request processors that will be used by the
         *        client. overrides the previous value. if set no default
         *        authentication processors will be automatically added. if set
         *        to <code>null</code> or not set at all the default
         *        authentication processors will be automatically added.
         * @return {@link Builder}
         */
        public Builder setRequestProcessors(List<RequestProcessor> processors) {
            // TODO the default proc javadoc should be moved to the authn
            // ProtocolFactory
            this.requestProcessors = processors;
            return this;
        }

        /**
         * @return the current request processor list
         */
        public List<RequestProcessor> getRequestProcessors() {
            return requestProcessors;
        }

        /**
         * Sets the executor the client must use to process requests.
         *
         * @param executor executor to be used by the client to process
         *        requests.
         * @return {@link Builder}
         */
        public Builder setExecutor(Executor executor) {
            this.executor = executor;
            return this;
        }

        /**
         * Sets if streaming (pseudo-push) responses will be requested by the
         * API client.
         * <p>
         * The default value is {@code true} to allow generated binding code and
         * newly written code using {@link ApiProvider} to consume stream APIs.
         * <p>
         * If there is client code making direct use of {@link ApiProvider} that
         * is not handling {@link MethodResult#getNext()} streaming can be
         * disabled to provide backwards compatibility. Turning off streaming
         * will not allow consumption of stream data. The client will receive
         * only the first chunk of stream data or error in such case.
         * <p>
         * When using generated bindings streaming should be enabled to assure
         * compatibility with new APIs.
         * <p>
         * Bindings generated with prior versions of the system cannot work with
         * streaming. There may also be other caveats and it is strongly
         * recommended to regenerate the bindings with the latest version
         * instead of switching this compatibility mode.
         *
         * @param if {@code true} the client will advertise stream support in
         *        HTTP {@code Accept} header otherwise the client will not
         *        advertise stream support.
         * @return the current {@link Builder} instance.
         */
        public Builder setStreamingEnabled(boolean enabled) {
            streamingEnabled = enabled;
            return this;
        }

        /**
         * Sets the {@link Tracer} for the vapi runtime.
         *
         * @param tracer the tracer to set.
         * @return the current {@link Builder} instance.
         */
        public Builder setTracer(Tracer tracer) {
            this.tracer = Objects.requireNonNull(tracer, "tracer");
            return this;
        }

        /**
         * @return immutable {@link ClientConfiguration}
         */
        public ClientConfiguration getConfig() {
            return new ClientConfiguration(requestProcessors,
                                           executor,
                                           streamingEnabled,
                                           tracer
                                           );
        }

    }
}
