/* **********************************************************
 * Copyright (c) 2012-2018, 2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.protocol.common.http;

import java.util.Arrays;
import java.util.List;

public final class HttpConstants {

    private HttpConstants() {
    }

    /**
     * The HTTP {@code Content-Type} header name.
     *
     * @see <a href="http://tools.ietf.org/html/rfc7231#section-3.1.1.5">Section
     *      3.1.1.5 of RFC 7231</a>
     */
    public static final String HEADER_CONTENT_TYPE = "Content-Type";

    /**
     * The HTTP {@code User-Agent} header name.
     *
     * @see <a href="http://tools.ietf.org/html/rfc7231#section-5.5.3">Section
     *      5.5.3 of RFC 7231</a>
     */
    public static final String HEADER_USER_AGENT = "User-Agent";

    /**
     * The HTTP {@code Accept} header name.
     *
     * @see <a href="http://tools.ietf.org/html/rfc7231#section-5.3.2">Section
     *      5.3.2 of RFC 7231</a>
     */
    public static final String HEADER_ACCEPT = "Accept";

    /**
     * The HTTP header name reserved for service identification.
     */
    public static final String HEADER_SERVICE_ID = "vapi-service";

    /**
     * The HTTP header name reserved for operation identification.
     */
    public static final String HEADER_OPERATION_ID = "vapi-operation";

    /**
     * The HTTP header name reserved for session identification.
     */
    public static final String HEADER_SESSION_ID = "vmware-api-session-id";

    /**
     * The HTTP header prefix for application context. This is used with vAPI
     * JSON-RPC 1.1 protocol.
     */
    public static final String HEADER_APPLICATION_CONTEXT_PREFIX = "vapi-ctx-";

    /**
     * Content-Type for JSON HTTP requests and single frame JSON HTTP responses.
     */
    public static final String CONTENT_TYPE_JSON = "application/json";

    /**
     * Content-Type prefix for HTTP responses which contain multiple response
     * frames.
     */
    public static final String CONTENT_TYPE_FRAMED = "application/vnd.vmware.vapi.framed";

    /** VAPI JSON-RPC with multiple streamed responses and type information */
    public static final String CONTENT_TYPE_STREAM_JSON = "application/vnd.vmware.vapi.stream.json";

    /** VAPI JSON-RPC without type information and framing */
    public static final String CONTENT_TYPE_CLEAN_JSON = "application/vnd.vmware.vapi.cleanjson";

    /** VAPI JSON-RPC with multiple streamed response frames without type information  */
    public static final String CONTENT_TYPE_CLEAN_STREAM_JSON = "application/vnd.vmware.vapi.clean.stream.json";

    /**
     * The HTTP {@code accept-language} header name.
     *
     * @see <a href="https://tools.ietf.org/html/rfc7231#section-5.3.5">Section
     *      5.3.5 of RFC 7231</a> Optional parameter following the format of the
     *      HTTP 1.1 Accept-Language header (RFC 7231 section 5.3.5). It is used to
     *      select a localization bundle. The selection process should roughly
     *      follow the guidelines of RFC 4647.
     *
     *      UI current use case involves one main locale with a single fallback.
     *      UI is used for sending locales in format like en_US, de_DE. Those
     *      need to be matched against bundle names that predominantly contain
     *      only language.
     *
     *      When missing, "en" should be used.
     */
    public static final String HEADER_ACCEPT_LANGUAGE = "accept-language";

    /**
     * The HTTP header names being used by vAPI's runtime.
     */
    public static final List<String> RESERVED_HEADERS = Arrays
            .asList(HEADER_CONTENT_TYPE,
                    HEADER_USER_AGENT,
                    HEADER_ACCEPT);
}
