/* **********************************************************
 * Copyright 2010 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.protocol.server.rpc.http;

import java.util.Map;

import javax.servlet.Servlet;

/**
 * Represents a service entry point.
 */

public interface Service {
   /**
    * Restrictions on the type of connection this service.
    */
   enum TransportGuarantee {
      /** Usually allows both HTTP and HTTPS*/
      NONE,
      /** Usually allows only HTTPS */
      INTEGRAL,
      /** Usually allows only HTTPS */
      CONFIDENTIAL
   }

   /**
    * Get the path under which the service can be accessed.
    *
    * @return a standard servlet mapping pattern.
    */
   String getPath();

   /**
    * Set the path under which the service can be accessed.
    * <p>
    * The path must follow the standard Servlet spec mapping patterns:
    * <ul>
    * <li><b>Exact</b>: starts with "/" and does not end with "/*".</li>
    * <li><b>Prefix</b>: starts with "/" and ends with "/*".</li>
    * <li><b>Extension</b>: starts with "*."</li>
    * <li><b>Default</b>: equals "/".</li>
    * </ul>
    * <p>
    * Request paths are tested against the mapping patterns in the standard
    * order:
    * <ol>
    * <li><b>Exact match</b>: E.g. "/foo".</li>
    * <li><b>Longest prefix match</b>: E.g. "/foo/bar/*" is tested before
    * "/foo/*".</li>
    * <li><b>Longest extension match</b>: E.g. "*.html" is tested before
    * "*.htm".</li>
    * <li><b>Default always matches</b>: E.g. use this if no other matches.</li>
    * </ol>
    * <p>
    * <b>Note</b><br>
    * For backward compatibility all exact patterns are also registered under a
    * path with a trailing "/". E.g. "/foo/bar/baz" is also registered under
    * "/foo/bar/baz/".
    *
    * @param path the path
    */
   void setPath(String path);

   /**
    * Get the Servlet that handles this service entry point.
    *
    * @return a Servlet object.
    */
   Servlet getServlet();

   /**
    * Set a Servlet to handle this service.
    *
    * @param servlet the Servlet.
    */
   void setServlet(Servlet servlet);

   /**
    * Get the parameters, which which the Servlet will be.
    *
    * @return a Map with the initial parameters or <code>null</code> if no
    *         initial configuration is set.
    */
   Map<String, String> getInitParameters();

   /**
    * Set the initial configuration of the servlet. These parameters will be
    * accessible to the Servlet via it's {@link javax.servlet.ServletConfig}.
    *
    * @param params the parameters.
    */
   void setInitParameters(Map<String, String> params);

   /**
    * Specify what connection guarantees this service expects from the
    * container.
    *
    * @param guarantee
    */
   void setTransportGuarantee(TransportGuarantee guarantee);

   /**
    * Obtain the connection restriction this service places on the container.
    *
    * @return the connection restriction
    */
   TransportGuarantee getTransportGuarantee();
}