/* **********************************************************
 * Copyright 2011-2013, 2019 VMware, Inc. All rights reserved.
 *      -- VMware Confidential
 * *********************************************************
 */

/*
 * ApiAggregator.java --
 *
 *      Implementation of an aggregating API provider.
 */

package com.vmware.vapi.provider.aggregator;

import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.core.ApiProvider;
import com.vmware.vapi.core.AsyncHandle;
import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.MethodResult;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.diagnostics.LogDiagnosticUtil;
import com.vmware.vapi.diagnostics.LogDiagnosticsConfigurator;
import com.vmware.vapi.diagnostics.Slf4jMDCLogConfigurator;
import com.vmware.vapi.internal.provider.introspection.IntrospectionFilter;
import com.vmware.vapi.internal.util.Validate;

/**
 * The <code>ApiAggregator</code> class is an aggregating
 * implementation of the <code>ApiProvider</code> interface.
 */
public class ApiAggregator implements ApiProvider {
    static final Logger logger = LoggerFactory.getLogger(ApiAggregator.class);

    /**
     * Default name of this provider.
     */
    public static final String AGGREGATOR_DEFAULT_NAME = "ApiAggregator";

    private final LogDiagnosticsConfigurator logDiag =
        new Slf4jMDCLogConfigurator();

    private final ApiProvider impl;

    /**
     * Calls {@link #ApiAggregator(String, Map, boolean)}, automatically
     * deploying introspection services.
     */
    public ApiAggregator(String name, Map<String, ApiProvider> ifaceMap) {
        this(name, ifaceMap, true);
    }

    /**
     * Constructor. Optionally can deploy a set of introspection services
     * which expose the introspection API of the provider through
     * {@link #invoke(String, String, DataValue, ExecutionContext, AsyncHandle)}.
     *
     * @param name  the name of the API aggregator
     *              if missing, {@link #AGGREGATOR_DEFAULT_NAME} will be used
     * @param ifaceMap map of service id to provider which handles that service
     * @param deployIntrospectionServices whether to automatically deploy
     *                                     introspection services
     */
    public ApiAggregator(String name, Map<String, ApiProvider> ifaceMap,
            boolean deployIntrospectionServices) {
        Validate.notNull(ifaceMap);
        Validate.noNullElements(ifaceMap.keySet());
        Validate.noNullElements(ifaceMap.values());
        ProviderAggregation core = new ProviderAggregation(
                getAggregatorName(name), ifaceMap);
        ApiProvider provider;
        if (deployIntrospectionServices) {
            provider = new IntrospectionFilter(core, core,
                    ProviderAggregation.AGGREGATOR_ERROR_DEFS);
        } else {
            provider = core;
        }
        this.impl = provider;
    }

    private static String getAggregatorName(String name) {
        if (name != null && !name.trim().isEmpty()) {
            name = name.trim();
        } else {
            name = AGGREGATOR_DEFAULT_NAME;
        }
        return name;
    }

    @Override
    public void invoke(String serviceId,
                       String operationId,
                       DataValue input,
                       ExecutionContext ctx,
                       AsyncHandle<MethodResult> asyncHandle) {
        try {
            logDiag.configureContext(LogDiagnosticUtil.getDiagnosticContext(ctx));
            impl.invoke(serviceId, operationId, input, ctx, asyncHandle);
        } catch (RuntimeException ex) {
            logger.error("Exception thrown in invokeMethod", ex);
            asyncHandle.setError(ex);
        } finally {
            logDiag.cleanUpContext(LogDiagnosticUtil.getDiagnosticKeys());
        }
    }
}
