/* **********************************************************
 * Copyright 2015, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.security;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import com.vmware.vapi.core.ExecutionContext.SecurityContext;
import com.vmware.vapi.internal.util.Validate;

/**
 * This class represents a security context suitable for OAuth2
 * authentication.
 */
public final class OAuthSecurityContext implements SecurityContext {

    public static final String ACCESS_TOKEN = "accessToken";

    private final Map<String, Object> properties = new HashMap<String, Object>();

    /**
     * @param accessToken Access token to be used for authentication, it cannot be null.
     */
    public OAuthSecurityContext(char[] accessToken) {
        Validate.notNull(accessToken);
        properties.put(ACCESS_TOKEN, accessToken);
        properties.put(AUTHENTICATION_SCHEME_ID, StdSecuritySchemes.OAUTH);
    }

    @Override
    public Object getProperty(String key) {
        return properties.get(key);
    }

    @Override
    public Map<String, Object> getAllProperties() {
        return Collections.unmodifiableMap(properties);
    }

    /**
     * Retrieve the access token that should be used for authentication.
     *
     * @return Access token to be used for authentication, it won't be null.
     */
    public char[] getAccessToken() {
        return (char[])properties.get(ACCESS_TOKEN);
    }

    /**
     * Creates a {@link OAuthSecurityContext} from the given
     * {@link SecurityContext} if it has an access token. Otherwise
     * <code>null</code> will be returned.
     *
     * @param context cannot be null.
     * @return the parsed {@link OAuthSecurityContext}.
     */
    public static OAuthSecurityContext newInstance(SecurityContext context) {
        Validate.notNull(context);
        Object accessTokenObj = context.getProperty(ACCESS_TOKEN);
        if (accessTokenObj instanceof String) {
            return new OAuthSecurityContext(
                    ((String) accessTokenObj).toCharArray());
        } else if (accessTokenObj instanceof char[]) {
            return new OAuthSecurityContext((char []) accessTokenObj);
        }
        return null;
    }
}
