/* **********************************************************
 * Copyright 2013, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.security;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import com.vmware.vapi.core.ExecutionContext.SecurityContext;
import com.vmware.vapi.internal.util.Validate;

/**
 * This class represents a security context suitable for user/password
 * authentication.
 */
public final class UserPassSecurityContext implements SecurityContext {

    public static final String USER_KEY = "userName";
    public static final String PASSWORD_KEY = "password";

    private final Map<String, Object> properties = new HashMap<String, Object>();

    /**
     * @param user cannot be null.
     * @param password cannot be null.
     */
    public UserPassSecurityContext(String user, char[] password) {
        Validate.notNull(user);
        Validate.notNull(password);
        properties.put(USER_KEY, user);
        properties.put(PASSWORD_KEY, password);
        properties.put(AUTHENTICATION_SCHEME_ID, StdSecuritySchemes.USER_PASS);
    }

    @Override
    public Object getProperty(String key) {
        return properties.get(key);
    }

    @Override
    public Map<String, Object> getAllProperties() {
        return Collections.unmodifiableMap(properties);
    }

    /**
     * @return the authenticating principal name. cannot be null.
     */
    public String getUser() {
        return (String)properties.get(USER_KEY);
    }

    /**
     * @return the authenticating principal password. cannot be null.
     */
    public char[] getPassword() {
        return (char[])properties.get(PASSWORD_KEY);
    }

    /**
     * Creates a {@link UserPassSecurityContext} from the given
     * {@link SecurityContext} if possible. Otherwise <code>null</code>
     * will be returned.
     *
     * @param context cannot be null.
     * @return the parsed {@link UserPassSecurityContext}.
     */
    public static UserPassSecurityContext getInstance(SecurityContext context) {
        Validate.notNull(context);
        String user = getAsString(context.getProperty(USER_KEY));
        String passString = getAsString(context.getProperty(PASSWORD_KEY));
        char[] pass = (passString == null)?
                getAsCharArr(context.getProperty(PASSWORD_KEY)) : passString.toCharArray();
        if (user != null && pass != null) {
            return new UserPassSecurityContext(user, pass);
        }

        return null;
    }

    /**
     * @param value can be null.
     * @return the given value as a String if possible. otherwise
     * <code>null</code> will be returned.
     */
    private static String getAsString(Object value) {
        if (value != null && value instanceof String) {
            return (String) value;
        }

        return null;
    }

    /**
     * @param value can be null.
     * @return the given value as a String if possible. otherwise
     * <code>null</code> will be returned.
     */
    private static char[] getAsCharArr(Object value) {
        if (value != null && value instanceof char[]) {
            return (char[]) value;
        }

        return null;
    }
}
