/* **********************************************************
 * Copyright 2011-2013, 2019 VMware, Inc. All rights reserved.
 *      -- VMware Confidential
 * *********************************************************
 */

/*
 * MessageFactory.java --
 *
 *      Factory for vAPI Messages
 */

package com.vmware.vapi;

import java.util.Arrays;
import java.util.Locale;
import java.util.ResourceBundle;

import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.l10n.StringFormatTemplateFormatter;
import com.vmware.vapi.l10n.TemplateFormatter;

/**
 * Factory for {@link Message} instances. This factory is intended to be used
 * for creating localizable message instances for dynamic vAPI providers or
 * the vAPI runtime itself.
 *
 * <p>Messages data (keys, templates) are loaded from {@link ResourceBundle}.
 */
public final class MessageFactory {
    private static MessageFactory defaultFactory;

    static {
        ResourceBundle runtimeBundle = ResourceBundle.getBundle("runtime");

        defaultFactory = new MessageFactory(runtimeBundle);
    }

    private ResourceBundle resBundle;
    private TemplateFormatter formatter;

    public MessageFactory(ResourceBundle resBundle,
                          TemplateFormatter formatter) {
        Validate.notNull(resBundle);
        Validate.notNull(formatter);

        this.resBundle = resBundle;
        this.formatter = formatter;
    }

    public MessageFactory(ResourceBundle resBundle) {
        this(resBundle, new StringFormatTemplateFormatter());
    }

    /**
     * Return a message object for the given id with the given args.
     *
     * @see #buildMessage(String, String...)
     */
    public static Message getMessage(String id, String... args) {
        return defaultFactory.buildMessage(id, args);
    }

    /**
     * Return a message object for the given id with the given args. The
     * {@link Message#getDefaultMessage()} of the result instance is formatted
     * and arguments values are replaced in.
     *
     * <p>If the message isn't found a default 'unknown' message is returned
     *
     * @param id identifier for the message
     * @param args values for placeholders in the message template
     * @return a <code>Message</code> instance; never <code>null</code>
     */
    public Message buildMessage(String id, String... args) {
        if (resBundle.containsKey(id)) {
            return new Message(id,
                              buildDefaultMessage(resBundle.getString(id),
                                                  resBundle.getLocale(),
                                                  args),
                              args);
        } else {
            return buildUnavailableMessage(id, args);
        }
    }

    String buildDefaultMessage(String msgTemplate, Locale locale, String... args) {
        if (locale == null) {
            locale = Locale.ENGLISH;
        }

        return formatter.format(msgTemplate,
                                Arrays.asList(args),
                                locale);
    }

    Message buildUnavailableMessage(String id, String... args) {
        return new Message("vapi.message.unknown",
                           String.format("Unknown message ID requested %s with parameters %s",
                                         id, Arrays.toString(args)));
    }
}
