/* **********************************************************
 * Copyright (c) 2012, 2019-2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.activation;

import java.util.Collections;
import java.util.UUID;

import com.vmware.vapi.bindings.client.InvocationConfig;
import com.vmware.vapi.bindings.server.AsyncContext;
import com.vmware.vapi.bindings.server.InvocationContext;
import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.ExecutionContext.ApplicationData;
import com.vmware.vapi.core.ExecutionContext.Builder;
import com.vmware.vapi.internal.util.Validate;

/**
 * Utility methods for working with activations.
 */
public final class Activations {

    /**
     * The name of the activation identifier property in an execution context.
     */
    public static final String PROP_NAME_ACTIVATION_ID = "ActivationId";

    private Activations() {
    }

    /**
     * Retrieves an activation identifier from the specified configuration.
     *
     * @param invocationConfig invocation configuration; can be
     *                         <code>null</code>
     * @return the activation identifier; <code>null</code> if no identifier
     */
    public static String getActivationId(InvocationConfig invocationConfig) {
        if (invocationConfig == null) {
            return null;
        }
        return getActivationId(invocationConfig.getExecutionContext());
    }

    /**
     * Retrieves an activation identifier from the specified context.
     *
     * @param asyncContext async context; can be <code>null</code>
     * @return the activation identifier; <code>null</code> if no identifier
     */
    public static <T> String getActivationId(AsyncContext<T> asyncContext) {
        if (asyncContext == null) {
            return null;
        }
        return getActivationId(asyncContext.getInvocationContext());
    }

    /**
     * Retrieves an activation identifier from the specified context.
     *
     * @param invocationContext invocation context; can be <code>null</code>
     * @return the activation identifier; <code>null</code> if no identifier
     */
    public static String getActivationId(InvocationContext invocationContext) {
        if (invocationContext == null) {
            return null;
        }
        return getActivationId(invocationContext.getExecutionContext());
    }

    /**
     * Retrieves an activation identifier from the specified context.
     *
     * @param executionContext execution context; can be <code>null</code>
     * @return the activation identifier; <code>null</code> if no identifier
     */
    public static String getActivationId(ExecutionContext executionContext) {
        if (executionContext == null) {
            return null;
        }
        return getActivationId(executionContext.retrieveApplicationData());
    }

    /**
     * Retrieves an activation identifier from the specified application data
     * object.
     *
     * @param applicationData application data; can be <code>null</code>
     * @return the activation identifier; <code>null</code> if no identifier
     */
    public static String getActivationId(ApplicationData applicationData) {
        if (applicationData == null)  {
            return null;
        }
        return applicationData.getProperty(PROP_NAME_ACTIVATION_ID);
    }

    /**
     * Generates a new activation identifier as a UUID.
     *
     * @return activation identifier
     */
    public static String newActivationId() {
        return UUID.randomUUID().toString();
    }

    /**
     * Creates a copy of the specified application data with the specified
     * activation identifier.
     *
     * @param appData application data; can be <code>null</code>
     * @param activationId activation identifier; must not be <code>null</code>
     * @return augmented application data
     */
    public static ApplicationData setActivationId(ApplicationData appData,
                                                  String activationId) {
        Validate.notNull(activationId);
        return ApplicationData.merge(appData, PROP_NAME_ACTIVATION_ID, activationId);
    }

    /**
     * Creates a new application data with the specified activation identifier.
     *
     * @param activationId activation identifier; must not be <code>null</code>
     * @return application data
     */
    public static ApplicationData newApplicationData(String activationId) {
        Validate.notNull(activationId);

        return new ApplicationData(Collections.singletonMap(
                PROP_NAME_ACTIVATION_ID, activationId));
    }

    /**
     * Creates a new application data with an activation identifier, generated
     * by {@link #newActivationId()}.
     *
     * @return application data
     */
    public static ApplicationData newApplicationData() {
        return newApplicationData(newActivationId());
    }


    /**
     * Creates a copy of the specified execution context with the specified
     * activation identifier.
     *
     * @param ctx execution context; can be <code>null</code>
     * @param activationId activation identifier; must not be <code>null</code>
     * @return execution context
     */
    public static ExecutionContext setActivationId(ExecutionContext ctx,
                                                   String activationId) {
        Validate.notNull(activationId);

        return Builder.from(ctx)
                .mergeApplicationData(PROP_NAME_ACTIVATION_ID, activationId)
                .build();
    }

    /**
     * Creates a new execution context with the specified activation
     * identifier.
     *
     * @param activationId activation identifier; must not be <code>null</code>
     * @return execution context
     */
    public static ExecutionContext newExecutionContext(String activationId) {
        return new ExecutionContext(newApplicationData(activationId), null);
    }

    /**
     * Creates a new execution context with an activation identifier, generated
     * by {@link #newActivationId()}.
     *
     * @return execution context
     */
    public static ExecutionContext newExecutionContext() {
        return newExecutionContext(newActivationId());
    }

    /**
     * Creates a copy of the specified invocation configuration with the
     * specified activation identifier.
     *
     * @param ctx invocation configuration; can be <code>null</code>
     * @param activationId activation identifier; must not be <code>null</code>
     * @return invocation configuration
     */
    public static InvocationConfig setActivationId(InvocationConfig ctx,
                                                   String activationId) {
        Validate.notNull(activationId);

        ExecutionContext executionContext = null;
        if (ctx != null) {
            executionContext = ctx.getExecutionContext();
        }

        return new InvocationConfig(
                setActivationId(executionContext, activationId));
    }

    /**
     * Creates a new invocation configuration with the specified activation
     * identifier.
     *
     * @param activationId activation identifier; must not be <code>null</code>
     * @return invocation configuration
     */
    public static InvocationConfig newInvocationConfig(String activationId) {
        return new InvocationConfig(newExecutionContext(activationId));
    }

    /**
     * Creates a new invocation configuration with an activation identifier,
     * generated by {@link #newActivationId()}.
     *
     * @return invocation configuration
     */
    public static InvocationConfig newInvocationConfig() {
        return newInvocationConfig(newActivationId());
    }


}
