/* **********************************************************
 * Copyright (c) 2012-2013, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.bindings;

import com.vmware.vapi.CoreException;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.StructValue;

/**
 * Base interface that represents a Java language bindings structure. <br>
 * <b>WARNING:</b> Users are not expected to implement this interface
 * because new methods might be added in the future.
 */
public interface Structure {

    /**
     * Returns this <code>Structure</code> represented in the dynamic
     * {@link DataValue} model.
     *
     * <p>Attempts to change the returned <code>StructValue</code>
     * representation are not required to change the state of this structure.
     * Actually the behavior is implementation dependent and is unspecified.
     * In general the returned value shouldn't be changed.
     *
     * @return <code>StructValue</code> representing this structure
     */
    StructValue _getDataValue();

    /**
     * Checks if the runtime type name of this structure matches the type
     * represented by the specified binding class.
     *
     * <p>This can be used as efficient check to determine if this instance
     * can be successfully {@link #_convertTo(Class) converted} to the target
     * type.
     *
     * @param clazz bindings class for the target structure; must not be {@code null}
     * @return true if this structures type name matches exactly the specified
     *         class' type name.
     */
    boolean _hasTypeNameOf(Class<? extends Structure> clazz);

    /**
     * Converts this structure into an instance of the provided class structure
     * if possible. A conversion will be possible if the requested structure's
     * fields are all present in this structure.
     *
     * @param clazz type of the result structure. cannot be null.
     * @return an instance of the requested type if conversion is possible.
     *         {@link CoreException} is thrown otherwise.
     */
    <T extends Structure> T _convertTo(Class<T> clazz);

    /**
     * Returns the canonical name of the structure. By default the canonical
     * name is derived from the fully-qualified IDL name by converting to lower
     * case and inserting underscores between words. For example
     * <code>org.example.FooBar</code> would turn into
     * <code>org.example.foo_bar</code>.
     *
     * @return canonical name
     */
    String _getCanonicalName();
}
