/* **********************************************************
 * Copyright (c) 2012, 2018 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.bindings.client;

import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.HttpResponseAccessor;
import com.vmware.vapi.core.ExecutionContext.RuntimeData;

/**
 * Configuration for a method invocation. If it contains a non-null security
 * context, it would override the security context from the stub configuration.
 */
public final class InvocationConfig {

   private final ExecutionContext executionContext;

   /**
    * Creates a configuration object which provides an execution context to a
    * method invocation.
    *
    * @param executionContext execution context; can be <code>null</code>
    */
   public InvocationConfig(ExecutionContext executionContext) {
      this.executionContext = executionContext;
   }

   /**
    * Returns the execution context associated with this configuration.
    *
    * @return execution context; can be <code>null</code>
    */
   public ExecutionContext getExecutionContext() {
      return executionContext;
   }

   /**
    * Static factory for instances with configured timeout as described in
    * {@link ExecutionContext.Builder#setReadTimeout} for one or more specific invocations.
    * <p>Usage:<br>
    * <code>
    * stub.operation(operationParam, InvocationConfig.readTimeout(1000));
    * </code></p>
    *
    * @param readTimeout non-negative; a value of zero is interpreted as an
    *        infinite timeout
    * @return a new instance configured with the specified read-timeout
    * @see com.vmware.vapi.core.ExecutionContext.Builder
    */
   public static InvocationConfig withReadTimeout(int readTimeout) {
      return new InvocationConfig(new ExecutionContext.Builder()
                                                      .setReadTimeout(readTimeout)
                                                      .build());
   }

   /**
    * Static factory for instances configured with {@link HttpResponseAccessor}.
    *
    * @param accessor which will be provided with the raw REST response
    * @return a new instance, configured with the specified accessor
    * @see com.vmware.vapi.core.ExecutionContext.Builder
    */
   public static InvocationConfig withResponseAccessor(HttpResponseAccessor accessor) {
       return new InvocationConfig(new ExecutionContext.Builder()
                                                       .setResponseAccessor(accessor)
                                                       .build());
   }

}
