/* **********************************************************
 * Copyright 2012-2014, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/

/*
 * PropertyConfigurator.java --
 *
 *      Creates vAPI runtime objects from a Java property file configuration.
 */

package com.vmware.vapi.config;

import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.core.ApiProvider;
import com.vmware.vapi.internal.ClassLoaderUtil;
import com.vmware.vapi.internal.protocol.ProtocolHandlerFactoryImpl;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.protocol.ProtocolHandler;
import com.vmware.vapi.protocol.ProtocolHandlerFactory;
import com.vmware.vapi.protocol.ProtocolHandlerFactory.KeyStoreConfig;
import com.vmware.vapi.protocol.ProtocolHandlerFactory.TrustStoreConfig;
import com.vmware.vapi.provider.ApiInterface;
import com.vmware.vapi.provider.local.LocalProvider;

/**
 * Creates vAPI runtime objects from a Java property file configuration.
 *
 * <p>
 * The format of the property file is as follows:
 * </p>
 *
 * <pre>
 *
 * #
 * # Primary API provider - mandatory
 * #
 * provider.type=[full name of provider class]
 * provider.name=[name of the provider]
 * local.interfaces=[list of full names of service classes]
 *
 * #
 * # Protocol handlers - mandatory
 * #
 * protocols=[list of protocol aliases]
 * protocol.[protocol alias].rpc=[rpc protocol alias]
 * protocol.[protocol alias].rpc.[rpc protocol alias].args=[arguments for non-ssl rpc endpoint]
 *
 * #
 * # SSL configuration for protocol handler - optional
 * #
 * protocol.[protocol alias].ssl=[true/false]
 * protocol.[protocol alias].ssl.keystore=[keystore for ssl]
 * protocol.[protocol alias].ssl.password=[password for the ssl keystore]
 * protocol.[protocol alias].ssl.keypassword=[password for the ssl key pair]
 * protocol.[protocol alias].rpc.[rpc protocol alias].secureargs=[arguments for ssl rpc endpoint]
 *
 * #
 * # SSL configuration for mutual authentication - optional
 * #
 * protocol.[protocol alias].ssl.client.auth=[true/want/false] true to require a
 *     valid certificate from the client to accept a connection and fail if the
 *     certificate is not supplied, want to request the client to supply a valid
 *     certificate, but do not fail if is not provided, false to not require a
 *     client certificate
 * protocol.[protocol alias].ssl.client.keystore=[client keystore for ssl]
 * protocol.[protocol alias].ssl.client.password=[password for the ssl client keystore]
 * protocol.[protocol alias].ssl.client.alias=[alias for the key to be used by the client]
 *
 * </pre>
 */
public final class PropertyConfigurator implements Configurator {

    private final Properties props;
    private final ProtocolHandlerFactory handlerFactory;
    private final ApiProvider provider;
    private final List<ProtocolHandler> handlers;
    static final Logger logger =
            LoggerFactory.getLogger(PropertyConfigurator.class);


    public PropertyConfigurator(Properties props) {
        this(props, new ProtocolHandlerFactoryImpl());
    }

    public PropertyConfigurator(Properties props,
                                ProtocolHandlerFactory handlerFactory) {
        Validate.notNull(props);
        Validate.notNull(handlerFactory);
        this.props = props;
        this.handlerFactory = handlerFactory;
        this.provider = createApiProvider("");
        this.handlers = createProtocolHandlers("", "protocols", provider);
    }

    @Override
    public ApiProvider getApiProvider() {
        return provider;
    }

    @Override
    public List<ProtocolHandler> getProtocolHandlers() {
        return handlers;
    }

    private ApiProvider createApiProvider(String propPrefix) {
        String providerName = getProperty(propPrefix, "provider.name");
        Class<? extends ApiProvider> providerClass =
                getProviderClass(propPrefix);
        if (LocalProvider.class == providerClass) {
            return createLocalProvider(providerName, propPrefix);
        }
        throw new IllegalArgumentException("Unsupported provider type: "
                + providerClass.getCanonicalName());
    }

    private Class<? extends ApiProvider> getProviderClass(String propPrefix) {
        String providerClassName = getRequiredProperty(propPrefix,
                "provider.type");
        Class<?> clazz;
        try {
            clazz = Class.forName(providerClassName);
        } catch (Exception ex) {
            throw new IllegalArgumentException("Could not load provider class "
                    + providerClassName, ex);
        }
        if (!ApiProvider.class.isAssignableFrom(clazz)) {
            throw new IllegalArgumentException("Provider class "
                    + providerClassName
                    + " does not implement ApiProvider interface ");
        }
        @SuppressWarnings("unchecked")
        Class<? extends ApiProvider> providerClass =
            (Class<? extends ApiProvider>) clazz;
        return providerClass;
    }

    private LocalProvider createLocalProvider(String name,
                                              String propPrefix) {
        List<ApiInterface> ifaces = createApiInterfaces(propPrefix,
                                                        "local.interfaces");
        return new LocalProvider(name, ifaces);
    }

    private List<ApiInterface> createApiInterfaces(String propPrefix,
                                                   String listPropName) {
        List<ApiInterface> ifaces = new ArrayList<ApiInterface>();
        String namesList = getProperty(propPrefix, listPropName);
        if (namesList == null) {
            return ifaces;
        }
        String[] names = namesList.split(",");
        for (String className : names) {
            ApiInterface i = loadInterfaceByClass(className);
            ifaces.add(i);
        }
        return ifaces;
    }

    private static ApiInterface loadInterfaceByClass(String className) {
        className = className.trim();
        try {
            ClassLoader cl = ClassLoaderUtil.getServiceClassLoader();
            Class<?> clazz = cl.loadClass(className);
            Class<? extends ApiInterface> ifaceClass = clazz
                    .asSubclass(ApiInterface.class);
            ApiInterface ifaceInstance = ifaceClass.newInstance();
            return ifaceInstance;
        } catch (Exception e) {
            throw new IllegalArgumentException(
                    "Unable to load service of class " + className, e);
        }
    }

    private List<ProtocolHandler> createProtocolHandlers(String propPrefix,
                                                         String listPropName,
                                                         ApiProvider provider) {
        List<ProtocolHandler> handlers = new ArrayList<ProtocolHandler>();
        String protocols = getProperty(propPrefix, listPropName);
        if (protocols == null || "".equals(protocols.trim())) {
            return handlers;
        }
        String[] protocolNames = protocols.split(",");
        for (String p : protocolNames) {
            String prefix = getPropertyName(propPrefix, "protocol." + p);
            ProtocolHandler handler = createProtocolHandler(prefix, provider);
            handlers.add(handler);
        }
        return handlers;
    }

    private ProtocolHandler createProtocolHandler(String propPrefix,
                                                  ApiProvider provider) {
        String rpcProtocol = getRequiredProperty(propPrefix, "rpc");
        String args = getProperty(propPrefix,
                                  "rpc." + rpcProtocol + ".args");
        String ssl = getProperty(propPrefix, "ssl");

        ProtocolHandler curr = null;
        if (ssl != null && ssl.equalsIgnoreCase("true")) {
            String secureArgs = getProperty(propPrefix,
                    "rpc." + rpcProtocol + ".secureargs");
            String keystore = getProperty(propPrefix, "ssl.keystore");
            String keystorePassword = getProperty(propPrefix, "ssl.password");
            String keyPassword = getProperty(propPrefix, "ssl.keypassword");
            String clientAuth =
                    getProperty(propPrefix, "ssl.client.auth", "false");
            String truststore = getProperty(propPrefix, "ssl.client.keystore");
            String truststorePassword =
                    getProperty(propPrefix, "ssl.client.password");
            KeyStoreConfig ksConfig =
                    new KeyStoreConfig(keystore,
                            keystorePassword,
                            keyPassword);
            TrustStoreConfig tsConfig =
                    new TrustStoreConfig(truststore, truststorePassword);
            curr = handlerFactory.getProtocolHandler(rpcProtocol, args,
                    secureArgs, clientAuth, ksConfig,
                    tsConfig, provider);
        } else if (ssl == null || ssl.equalsIgnoreCase("false")) {
            curr = handlerFactory.getInsecureProtocolHandler(rpcProtocol,
                    args, provider);
        } else {
            throw new IllegalArgumentException("SSL property '"
                    + getPropertyName(propPrefix, "ssl")
                    + "' has invalid value '" + ssl + "'");
        }
        return curr;
    }

    static String getPropertyName(String prefix,
                                  String name) {
        if (prefix != null && !prefix.isEmpty()) {
            return prefix + "." + name;
        }
        return name;
    }

    private String getRequiredProperty(String propPrefix,
                                       String propName) {
        String name = getPropertyName(propPrefix, propName);
        String value = props.getProperty(name, null);
        if (value == null || value.trim().isEmpty()) {
            throw new IllegalArgumentException("Missing property "+name);
        }
        return value;
    }

    private String getProperty(String propPrefix,
                               String propName) {
        return getProperty(propPrefix, propName, null);
    }

    private String getProperty(String propPrefix,
                               String propName,
                               String defaultValue) {
        return props.getProperty(getPropertyName(propPrefix, propName),
                                 defaultValue);
    }

}
