/* **********************************************************
 * Copyright 2012-2013, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.core;

import java.util.Set;

import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.internal.provider.introspection.ApiIntrospectionStubImpl;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.provider.introspection.ApiIntrospectionStub;

/**
 * Utility which provides synchronous method overloads on top of an
 * <code>ApiProvider</code>.
 */
public class ApiProviderStubImpl implements ApiProviderStub {
    private final ApiProvider provider;
    private final ApiIntrospectionStub introspection;

    public ApiProviderStubImpl(ApiProvider provider) {
        Validate.notNull(provider);
        this.provider = provider;
        this.introspection = new ApiIntrospectionStubImpl(provider);
    }

    @Override
    public void getDefinition(ExecutionContext ctx,
            AsyncHandle<ProviderDefinition> asyncHandle) {
        introspection.getDefinition(ctx, asyncHandle);
    }

    @Override
    public void getInterfaceIdentifiers(ExecutionContext ctx,
            AsyncHandle<Set<InterfaceIdentifier>> asyncHandle) {
        introspection.getInterfaceIdentifiers(ctx, asyncHandle);
    }

    @Override
    public void getInterface(ExecutionContext ctx,
            InterfaceIdentifier iface,
            AsyncHandle<InterfaceDefinition> asyncHandle) {
        introspection.getInterface(ctx, iface, asyncHandle);
    }

    @Override
    public void getMethod(ExecutionContext ctx,
            MethodIdentifier method,
            AsyncHandle<MethodDefinition> asyncHandle) {
        introspection.getMethod(ctx, method, asyncHandle);
    }

    @Override
    public void invoke(String serviceId,
                       String operationId,
                       DataValue input,
                       ExecutionContext ctx,
                       AsyncHandle<MethodResult> asyncHandle) {
        provider.invoke(serviceId, operationId, input, ctx, asyncHandle);
    }

    @Override
    public ProviderDefinition getDefinition() {
        return introspection.getDefinition();
    }

    @Override
    public Set<InterfaceIdentifier> getInterfaceIdentifiers() {
        return introspection.getInterfaceIdentifiers();
    }

    @Override
    public InterfaceDefinition getInterface(InterfaceIdentifier iface) {
        return introspection.getInterface(iface);
    }

    @Override
    public MethodDefinition getMethod(MethodIdentifier method) {
        return introspection.getMethod(method);
    }

    @Override
    public ProviderDefinition getDefinition(ExecutionContext ctx) {
        return introspection.getDefinition(ctx);
    }

    @Override
    public Set<InterfaceIdentifier> getInterfaceIdentifiers(
            ExecutionContext ctx) {
        return introspection.getInterfaceIdentifiers(ctx);
    }

    @Override
    public InterfaceDefinition getInterface(ExecutionContext ctx,
            InterfaceIdentifier iface) {
        return introspection.getInterface(ctx, iface);
    }

    @Override
    public MethodDefinition getMethod(ExecutionContext ctx,
            MethodIdentifier method) {
        return introspection.getMethod(ctx, method);
    }

    /**
     * Synchronous variant of
     * {@link #invoke(String, String, DataValue, ExecutionContext, AsyncHandle)}
     */
    @Override
    public MethodResult invoke(String serviceId,
                               String operationId,
                               DataValue input,
                               ExecutionContext ctx) {
        AsyncHandleSyncAdapter<MethodResult> asyncHandle =
            new AsyncHandleSyncAdapter<MethodResult>();
        provider.invoke(serviceId, operationId, input, ctx, asyncHandle);
        return asyncHandle.get();
    }
}
