/* **********************************************************
 * Copyright (c) 2012-2013, 2016 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.data;

import static com.vmware.vapi.MessageFactory.getMessage;

import java.util.Arrays;

import com.vmware.vapi.CoreException;

public final class BlobValue implements DataValue {

    private static final long serialVersionUID = 1L;
    private final byte[] value;

    /**
     * @param value must not be <code>null</code>. note that the value will NOT
     *              be copied.
     */
    public BlobValue(byte[] value) {
        if (value == null) {
            throw new CoreException(getMessage("vapi.data.blob.null.bytearray"));
        }

        this.value = value;
    }

    @Override
    public DataType getType() {
        return DataType.BLOB;
    }

    /**
     * @return the byte array that represents this BLOB. must not be
     * <code>null</code>
     */
    public byte[] getValue() {
        return this.value;
    }

    @Override
    public boolean equals(Object o) {
        if (o instanceof BlobValue) {
            BlobValue blob = (BlobValue) o;
            return Arrays.equals(value, blob.value);
        }
        return false;
    }

    @Override
    public int hashCode() {
        return Arrays.hashCode(value);
    }

    @Override
    public String toString() {
        return "<array of " + value.length + " bytes>";
    }

    @Override
    public void accept(ValueVisitor visitor) {
        visitor.visit(this);
    }

    @Override
    public BlobValue copy() {
        return new BlobValue(Arrays.copyOf(value, value.length));
    }
}
