/* **********************************************************
 * Copyright (c) 2017 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal;

import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Utility for accessing vAPI version information at runtime.
 *
 * <p>This util loads couple of classpath resources with version information
 * <ul><li><b>vapi_runtime_version.properties</b> - provided by the runtime itself</li>
 *     <li><b>vapi_sdk_version.properties</b> -  may be provided by consumers
 *            of the vAPI runtime, e.g. an SDK or dynamic vAPI client built on
 *            top of it</li>
 * </ul>
 *
 * Each of these resource must be a properties file with version information
 * about particular layer/component of the software stack (like runtime or SDK).
 * It must contain the following keys:
 * <ul><li><b>component.name</b> - name of the component (e.g. "vAPI" for
 *                                 the runtime)</li>
 *     <li><b>component.version</b> - version (e.g. "2.7.1")</li>
 *     <li><b>component.build</b> - build number</li>
 * </ul>
 *
 * The values of for these keys must be ASCII characters only, and should not
 * contain whitespace characters.
 */
public class VersionUtil {
    private static final Logger LOGGER =
            LoggerFactory.getLogger(VersionUtil.class);

    private static final String RUNTIME_VERSION_RESOURCE_NAME =
            "vapi_runtime_version.properties";
    private static final String SDK_VERSION_RESOURCE_NAME =
            "vapi_sdk_version.properties";

    private static final String USER_AGENT = buildUserAgent();

    /**
     * Returns a {@code String} representing the user agent.
     *
     * <p>The user agent string is built based on the version resources
     * loaded by this class. Examples:
     * <ul>
     *   <li>vAPI/2.7.1 Java/1.8.0_73 (Windows 10; 10.0; amd64)</li>
     *   <li>vAPI/2.7.1 Java/1.7.0_76 (Linux; 2.6.32-642.6.2.el6.x86_64; amd64)</li>
     *   <li>SDK/6.6.2 vAPI/2.7.1 Java/1.8.0_73 (Windows 10; 10.0; amd64)</li>
     *   <li>vRO/7.2.0 vAPI/2.7.1 Java/1.8.0_73 (Windows 10; 10.0; amd64)</li>
     * </ul>
     *
     * @return the user agent string
     */
    public static String getUserAgent() {
        return USER_AGENT;
    }

    static String buildUserAgent() {
        StringBuilder builder = new StringBuilder();
        buildUserAgentComponent(loadSdkVersion(), builder);
        buildUserAgentComponent(loadRuntimeVersion(), builder);
        buildJavaComponent(builder);
        buildOsComment(builder);
        return builder.toString();
    }

    static void buildUserAgentComponent(Properties verProperties,
                                        StringBuilder builder) {
        if (verProperties == null) {
            return;
        }

        builder.append(verProperties.getProperty("component.name"))
        .append('/')
        .append(verProperties.getProperty("component.version"))
        .append(' ');
    }

    static void buildJavaComponent(StringBuilder builder) {
        builder.append("Java/").append(System.getProperty("java.version"));
    }

    static void buildOsComment(StringBuilder builder) {
        builder.append(" (")
               .append(System.getProperty("os.name")).append("; ")
               .append(System.getProperty("os.version")).append("; ")
               .append(System.getProperty("os.arch")).append(")");
    }

    static Properties loadRuntimeVersion() {
        return loadVersionResource(RUNTIME_VERSION_RESOURCE_NAME);
    }

    static Properties loadSdkVersion() {
        return loadVersionResource(SDK_VERSION_RESOURCE_NAME);
    }

    static Properties loadVersionResource(String resourceName) {
        ClassLoader runtimeLoader = VersionUtil.class.getClassLoader();
        try (InputStream inStream = runtimeLoader.getResourceAsStream(resourceName)) {
            if (inStream == null) {
                return null;
            }

            Properties versionProps = new Properties();
            versionProps.load(inStream);
            return versionProps;
        } catch (IOException | IllegalArgumentException ex) {
            LOGGER.debug("Failed to load content of version info resource ", ex);
            return null;
        }
    }
}
