/* **********************************************************
 * Copyright (c) 2012-2015, 2017, 2019-2020, 2022 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.bindings;

import java.lang.reflect.Method;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Set;
import java.util.TimeZone;

import com.vmware.vapi.CoreException;
import com.vmware.vapi.Message;
import com.vmware.vapi.MessageFactory;
import com.vmware.vapi.bindings.DynamicStructure;
import com.vmware.vapi.bindings.DynamicStructureImpl;
import com.vmware.vapi.bindings.StaticStructure;
import com.vmware.vapi.bindings.Structure;
import com.vmware.vapi.bindings.convert.ConverterException;
import com.vmware.vapi.bindings.type.StructType;
import com.vmware.vapi.bindings.type.StructType.FieldNameDetails;
import com.vmware.vapi.bindings.type.Type;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.SecretValue;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.internal.bindings.convert.impl.GregorianCalendarConverter;
import com.vmware.vapi.internal.bindings.convert.impl.JavaUtilCalendarRfc3339DateTimeConverter;
import com.vmware.vapi.internal.util.Validate;

/**
 * Utility methods for implementation of
 * <code>equals()/hashCode()/toString()/extractBindingType()</code> for
 * generated Java bindings classes for vAPI structures.
 *
 * <p>This class is designed to be invoked from generate code for Java
 * language bindings.
 */
public final class BindingsUtil {

    /** converter between API runtime types and bindings types */
    private static final TypeConverter converter = new TypeConverterImpl();

    private static final GregorianCalendarConverter calendarConverter =
            new JavaUtilCalendarRfc3339DateTimeConverter();

    private static final String SINGLE_INDENT = "    ";

    /** Used for quick toString conversion for calendar objects */
    private static final ThreadLocal<SimpleDateFormat> dateTimeFormatter =
            new ThreadLocal<SimpleDateFormat>() {
                @Override
                protected SimpleDateFormat initialValue() {
                    return new SimpleDateFormat("M/d/yy HH:mm:ss Z");
                }
            };

    private static final TimeZone defaultTimeZone = TimeZone.getDefault();

    private BindingsUtil() {
    }

    /**
     * Compares two object representing vAPI structures in the Java language
     * bindings for equality. The values of all structures' fields + the
     * structures' names are taken into account when performing the comparison.
     *
     * <p>This method first converts the objects to <code>StructValue<code>
     * and then compares the result.
     *
     * @param leftStruct    instance to compare; must not be null
     * @param rightStruct   instance to compare with
     *
     * @return true if objects are equal, otherwise false;
     */
    public static boolean areEqual(StaticStructure leftStruct,
                                   Object rightStruct) {
        if (leftStruct == null) {
            return rightStruct == null;
        }

        if (leftStruct == rightStruct) {
            return true;
        }

        if (rightStruct == null) {
            // Sanity check, if the right is null we can't invoke .getClass()
            return false;
        }

        if (leftStruct.getClass() != rightStruct.getClass()) {
            return false;
        }

        StaticStructure staticRightStruct = (StaticStructure) rightStruct;

        // compare static fields
        for (String field : leftStruct._getType().getFieldNames()) {
            Method getter = leftStruct._getType().getGetterMethodForField(field);
            Object left = invokeGetter(getter, leftStruct);
            Object right = invokeGetter(getter, staticRightStruct);
            if (!compareStructureField(left, right)) {
                return false;
            }
        }

        // compare dynamic fields
        Set<String> leftDynamicFieldNames = leftStruct._getDynamicFieldNames();
        Set<String> rightDynamicFields = staticRightStruct._getDynamicFieldNames();
        if (!leftDynamicFieldNames.equals(rightDynamicFields)) {
            return false;
        }

        for (String fieldName : leftDynamicFieldNames) {
            DataValue leftValue = leftStruct._getDynamicField(fieldName);
            DataValue rightValue = staticRightStruct._getDynamicField(fieldName);
            if (!leftValue.equals(rightValue)) {
                return false;
            }
        }

        return true;
    }

    /**
     * Compares two objects. There is requirement left and right to be from the
     * same type (class).
     *
     * @param left the left object
     * @param right the right object
     * @return true iff the objects are equals
     */
    private static boolean compareStructureField(Object left, Object right) {
        if (left == null ^ right == null) {
            return false;
        }

        if (left == right) {
            return true;
        }

        if (left.getClass() == char[].class) {
            if (right.getClass() != char[].class ||
                !Arrays.equals((char[])left, (char[])right)) {
                return false;
            }
        } else if (left.getClass() == byte[].class) {
            if (right.getClass() != byte[].class ||
                !Arrays.equals((byte[])left, (byte[])right)) {
                return false;
            }
        } else if (left instanceof Calendar) {
            if (!calendarConverter.toValue(left).equals(
                    calendarConverter.toValue(right))) {
                return false;
            }
        } else if (left instanceof List) {
            List<?> leftList = (List<?>) left;
            List<?> rightList = (List<?>) right;
            if (leftList.size() != rightList.size()) {
                return false;
            }

            for (int i = 0; i < leftList.size(); i++) {
                if (!compareStructureField(leftList.get(i), rightList.get(i))) {
                    return false;
                }
            }
        } else if (left instanceof Map<?, ?>) {
            Map<?, ?> leftMap = (Map<?, ?>) left;
            Map<?, ?> rightMap = (Map<?, ?>) right;

            // Key set are URI, int or strings - so .equals should work
            if (!leftMap.keySet().equals(rightMap.keySet())) {
                return false;
            }

            Iterator<?> keyIterator = leftMap.keySet().iterator();
            while (keyIterator.hasNext()) {
                Object key = keyIterator.next();
                if (!compareStructureField(leftMap.get(key),
                                           rightMap.get(key))) {
                    return false;
                }
            }
        } else if (!left.equals(right)) {
            return false;
        }

        return true;
    }

    /**
     * Compute the hashCode of binding field.
     *
     * @param value the binding field value
     * @return hash code of the value
     */
    private static int generateHashCode(Object value) {
        if (value == null) {
            return 0;
        }

        int hashCode = 1;

        if (value.getClass() == char[].class) {
            hashCode = Arrays.hashCode((char[])value);
        } else if (value.getClass() == byte[].class) {
            hashCode = Arrays.hashCode((byte[])value);
        } else if (value instanceof Calendar) {
            hashCode = calendarConverter.toValue(value).hashCode();
        } else if (value instanceof List) {
            List<?> list = (List<?>) value;
            for (int i = 0; i < list.size(); i++) {
                hashCode = 31 * hashCode + generateHashCode(list.get(i));
            }
        } else if (value instanceof Map<?, ?>) {
            Map<?, ?> map = (Map<?, ?>) value;

            for (Entry<?, ?> e : map.entrySet()) {
                hashCode += Objects.hashCode(e.getKey())
                        ^ generateHashCode(e.getValue());
            }
        } else {
            hashCode = 31 * hashCode + value.hashCode();
        }

        return hashCode;
    }

    /**
     * Computes hash code for an object representing vAPI structures in
     * the Java language bindings. The values of all structure's fields + the
     * structure name are taken into account when performing the computation.
     *
     * <p>This method first converts the object to <code>StructValue<code>
     * and then computes the result.
     *
     * @param struct        instance to compute hash code for; must not be null
     * @return the computed hash code
     */
    public static int computeHashCode(StaticStructure struct) {
        if (struct == null) {
            throw new NullPointerException();
        }

        StructType type = struct._getType();
        int hashCode = type.getName().hashCode();

        // hashcode for static fields
        for (String field : type.getFieldNames()) {
            Method getter = type.getGetterMethodForField(field);
            Object value = invokeGetter(getter, struct);
            hashCode += field.hashCode()
                    ^ Objects.hashCode(generateHashCode(value));
        }

        // hashcode for dynamic fields
        for (String fieldName : struct._getDynamicFieldNames()) {
            hashCode += Objects.hashCode(fieldName)
                    ^ Objects.hashCode(struct._getDynamicField(fieldName));
        }

        return hashCode;
    }

    /**
     * Invokes a getter and return its result.
     *
     * @param getter the getter to be invoked
     * @param struct the structure that the getter should be invoked on
     * @return the getter result
     * @throws CoreException in case of error
     */
    private static Object invokeGetter(Method getter, StaticStructure struct) {
        try {
            return getter.invoke(struct);
        } catch (Exception e) {
            throw new CoreException(
                        "vapi.bindings.structbinding.struct.getter.error",
                        getter.getName(),
                        struct.getClass().getCanonicalName());
        }
    }

    /**
     * Builds <code>String</code> representation for a {@link Structure}
     * instance.
     *
     * <p>This method first converts the object to <code>StructValue<code>
     * and then computes the result.
     *
     * @param struct        instance to represent as string; must not be null
     * @param dynamicFields structValue containing all extra fields and possibly
     *                      some fields that are part of the static structure;
     *                      may be {@code null}
     * @return the built string representation
     */
    public static String convertToString(StaticStructure struct,
                                         StructValue dynamicFields) {
        return convertToString(struct, dynamicFields, 1);
    }

    static String convertToString(StaticStructure struct,
                                  StructValue dynamicFields,
                                  int indent) {
        Validate.notNull(struct);
        Validate.isTrue(indent >= 0);

        StructType type = struct._getType();

        StringBuilder result = new StringBuilder();
        result.append(struct.getClass().getSimpleName());
        result.append(" (").append(type.getName()).append(")");
        result.append(" => {\n");
        boolean isFirst = true;
        // print all static fields of the structure
        for (String field : type.getFieldNames()) {
            FieldNameDetails fnd = type.getFieldNameDetails(field);
            Method getter = type.getGetterMethodForField(field);
            Object value = invokeGetter(getter, struct);

            if (isFirst) {
                isFirst = false;
            } else {
                result.append(",\n");
            }

            writeIndentation(indent, result);
            result.append(fnd.getMixedCaseName()).append(" = ");
            appendObject(value, indent + 1, result);
        }

        // print any extra dynamic fields
        if (dynamicFields != null) {
            printDynamicFields(dynamicFields, indent, type, result);
        }

        // } - close the whole structure
        result.append("\n");
        writeIndentation(indent - 1, result);
        result.append("}");

        return result.toString();
    }

    private static void printDynamicFields(StructValue dynamicFields,
                                           int indent,
                                           StructType type,
                                           StringBuilder output) {
        boolean isFirst = true;
        for (String field : dynamicFields.getFieldNames()) {
            if (type.getFieldNames().contains(field)) {
                // static field; already printed
                continue;
            }

            if (isFirst) {
                output.append("\n");
                writeIndentation(indent, output);
                output.append("[dynamic fields]: {\n");
                isFirst = false;
            } else {
                output.append(",\n");
            }

            writeIndentation(indent + 1, output);
            output.append(field);
            output.append(" = ");
            output.append(dynamicFields.getField(field));
        }

        // close [dynamic fields]: section if it was started
        if (!isFirst) {
            output.append("\n");
            writeIndentation(indent, output);
            output.append("}");
        }
    }

    private static void writeIndentation(int indent, StringBuilder output) {
        for (int i = 0; i < indent; i++) {
            output.append(SINGLE_INDENT);
        }
    }

    /**
     * Handles object toString conversion appending the result to the specified
     * output. Handles Calendar with user friendly formatting, secret by
     * printing <secret>, binary fields (byte[]) by their length and the other
     * type of binding fields just with .toString() conversion.
     *
     * @param value     the value to convert
     * @param indent    required indentation
     * @param output    the output that will handle the result
     */
    private static void appendObject(Object value,
                                     int indent,
                                     StringBuilder output) {
        Validate.isTrue(indent >= 0);

        if (value == null) {
            output.append("<null>");
        } else if (value instanceof Calendar) {
            appendCalendar((Calendar) value, output);
        } else if (value.getClass() == char[].class) {
            // secret value, process it the same way as the SecretValue
            output.append(SecretValue.STRING_REPRESENTATION);
        } else if (value.getClass() == byte[].class) {
            // byte array, process it the same way as the BlobValue
            output.append("<array of ");
            output.append(((byte[])value).length);
            output.append(" bytes>");
        } else if (value instanceof StaticStructure) {
            StaticStructure staticStruct = (StaticStructure) value;
            output.append(convertToString(staticStruct,
                                          staticStruct._getDataValue(),
                                          indent));
        } else {
            output.append(value.toString());
        }
    }

    private static void appendCalendar(Calendar cal, StringBuilder output) {
        TimeZone targetZone = null;
        if (cal.getTimeZone() != null) {
            // format in the TimeZone of the Calendar instance (instead of
            // just using the default time zone - that will give the
            // same string even in different zones
            targetZone = cal.getTimeZone();
        } else {
            targetZone = defaultTimeZone;
        }

        dateTimeFormatter.get().setTimeZone(targetZone);
        output.append(dateTimeFormatter.get().format(cal.getTime()));
    }

    /**
     * Checks if the structure value has the type name of the specified
     * structure binding class. Returns <code>false</code> if the structure
     * binding class is a {@link DynamicStructure}.
     *
     * @param structValue structure value; must not be <code>null</code>
     * @param targetBindingClass structure binding class; must not be
     *                           <code>null</code>
     * @return whether the value has the same type name as the binding
     */
    public static boolean hasTypeNameOf(StructValue structValue,
                                        Class<? extends Structure> targetBindingClass) {
        Validate.notNull(structValue);
        Validate.notNull(targetBindingClass);
        if (DynamicStructure.class.isAssignableFrom(targetBindingClass)) {
            return false;
        }
        if (!StaticStructure.class.isAssignableFrom(targetBindingClass)) {
            throw new IllegalArgumentException(String.format(
                    "Unknown structure binding class '%s'",
                    targetBindingClass.getCanonicalName()));
        }
        @SuppressWarnings("unchecked")
        Class<? extends StaticStructure> staticStructClass =
                (Class<? extends StaticStructure>) targetBindingClass;
        StructType targetType = extractBindingType(staticStructClass);

        return matchesStructValueName(structValue, targetType) ||
                matchesDiscriminatorValue(structValue, targetType);
    }

    static boolean matchesStructValueName(StructValue structValue,
                                          StructType targetType) {
        return targetType.getName().equals(structValue.getName());
    }

    static boolean matchesDiscriminatorValue(StructValue structValue,
                                             StructType targetType) {
        String discrBy = targetType.getDiscriminatedBy();
        if (discrBy == null) {
            return false;
        }

        String discriminator = targetType.getDiscriminator();
        if (!structValue.hasField(discriminator)) {
            return false;
        }

        return discrBy.equals(structValue.getString(discriminator));
    }

    /**
     * Converts the <code>struct</code> structure into an instance of the
     * provided class structure if possible. The conversion will be possible
     * if the requested structure's fields are all present in the
     * <code>struct</code> instance.
     *
     * @param struct the structure that will be used as a source for the
     *        conversion. cannot be null.
     * @param clazz type of the result structure. cannot be null.
     * @param <T> type of the result structure
     * @return an instance of the requested type if conversion is possible.
     *         {@link CoreException} is thrown otherwise.
     */
    public static <T extends Structure> T convertTo(Structure struct,
                                                    Class<T> clazz) {
        return convertTo(struct, clazz, null);
    }

    public static <T extends Structure> T convertTo(Structure struct,
                                                    Class<T> clazz,
                                                    TypeConverter converter) {
        Validate.notNull(struct);
        Validate.notNull(clazz);
        if (clazz == DynamicStructure.class) {
            return clazz.cast(new DynamicStructureImpl(struct._getDataValue(), converter));
        } else if (StaticStructure.class.isAssignableFrom(clazz)) {
            @SuppressWarnings("unchecked")
            Type targetType = extractBindingType((Class<StaticStructure>) clazz);
            if (converter == null) {
                converter = new TypeConverterImpl();
            }
            Structure staticStructure = converter.convertToJava(struct._getDataValue().copy(),
                                                                targetType);
            return clazz.cast(staticStructure);
        } else {
            throw new IllegalArgumentException("Unknown structure type " + clazz.getName());
        }
    }

    /**
     * @param targetClass cannot be null
     * @param <T> type of the static structure
     * @return returns the binding type of the provided targetClass
     * @throws IllegalArgumentException if the specified class is not a binding
     */
    public static <T extends StaticStructure> StructType extractBindingType(Class<T> targetClass) {
        Validate.notNull(targetClass);
        StructType structType = null;
        try {
            structType = (StructType)
                    targetClass.getMethod("_getClassType").invoke(null);
        } catch (Exception e) {
            throw new IllegalArgumentException(
                    "Can not extract type information from non-binding object.",
                    e);
        }
        if (structType == null) {
            throw new IllegalArgumentException(
                    String.format(
                            "Structure binding class '%s' provides no type information",
                            targetClass.getCanonicalName()));
        }
        return structType;
    }

    /**
     * Converts Java object to a DataValue following the given type specification
     *
     * @param object the java object to be converted. can be null if the
     *               specified data type allows it (e.g. <code>OptionalType</code>)
     * @param type   the type specification of the resulting DataValue. cannot
     *               be null
     * @return DataValue which represents the specified Java object
     */
    public static DataValue toDataValue(Object object, Type type) {
        return converter.convertToVapi(object, type);
    }

    /**
     * Retrieves a field value from the structure.
     *
     * @param structure  the structure that contains the field to be get
     * @param fieldName  name of a field of the structure
     * @return field value
     * @throw ConverterException if the field cannot be retrieved
     */
    public static Object getStructureFieldValue(StaticStructure structure,
                                                StructType declaredType,
                                                String fieldName) {
        Method getterMethod = declaredType.getGetterMethodForField(fieldName);
        try {
            return getterMethod.invoke(structure);
        } catch (Exception ex) {
            Message msg = MessageFactory.getMessage(
                    "vapi.bindings.structbinding.struct.getter.error",
                    getterMethod.getName(),
                    structure.getClass().getCanonicalName());
            throw new ConverterException(msg, ex);
        }
    }

    /**
     * Sets a field on the structure.
     *
     * @param structure the structure binding instance to set the field to
     * @param declaredType type of the structure
     * @param fieldName name of the field to be set
     * @param fieldValue value of the field
     * @throws ConverterException if the field cannot be set, because respective setter can not
     *                            be found or invoked reflectively
     */
    public static void setStructureFieldValue(StaticStructure structure,
                                              StructType declaredType,
                                              String fieldName,
                                              Object fieldValue) {
        Method setterMethod = declaredType.getSetterMethodForField(fieldName);
        try {
            setterMethod.invoke(structure, fieldValue);
        } catch (Exception ex) {
            String fieldValueClass = fieldValue != null ?
                    fieldValue.getClass().getCanonicalName() : "null";
            Message msg = MessageFactory.getMessage(
                    "vapi.bindings.structbinding.struct.set.field.error",
                    setterMethod.getName(),
                    structure.getClass().getCanonicalName(),
                    fieldValueClass);
            throw new ConverterException(msg, ex);
        }
    }
}
