/* **********************************************************
 * Copyright (c) 2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.bindings;

import java.util.Collection;

import com.vmware.vapi.bindings.type.Type;
import com.vmware.vapi.core.AsyncHandle;
import com.vmware.vapi.core.Consumer;
import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.MethodResult;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.StructValue;

/**
 * {@code AsyncHandle} decorator which can retry the invocation if an error occurs.
 *
 * @param <T> type of the operation result
 */
abstract class RetryingHandle<T> extends ResultTranslatingHandle<T> {
    private final ResultTranslatingHandle<T> decoratedHandle;
    private final String serviceId;
    private final String operationId;
    private final ExecutionContext executionContext;
    private final StructValue inputValue;
    private final int invocationAttempt;

    RetryingHandle(Stub stub,
                   Type outputType,
                   Collection<Type> errorTypes,
                   ResultTranslatingHandle<T> decoratedHandle,
                   String serviceId,
                   String operationId,
                   ExecutionContext executionContext,
                   StructValue inputValue,
                   int invocationAttempt) {
        super(stub, outputType, errorTypes);
        this.decoratedHandle = decoratedHandle;
        this.serviceId = serviceId;
        this.operationId = operationId;
        this.executionContext = executionContext;
        this.inputValue = inputValue;
        this.invocationAttempt = invocationAttempt;
    }

    @Override
    public void updateProgress(DataValue progress) {
        decoratedHandle.updateProgress(progress);
    }

    @Override
    void onSuccess(T result, Consumer<AsyncHandle<MethodResult>> next) {
        decoratedHandle.onSuccess(result, next);
    }

    @Override
    void onFailure(RuntimeException error) {
        ExecutionContext ctx = stub.getRetryContext(serviceId,
                                                      operationId,
                                                      executionContext,
                                                      inputValue,
                                                      error,
                                                      invocationAttempt);
        if (ctx == null) {
            decoratedHandle.onFailure(error);
            return;
        }

        onRetry(ctx);
    }

    /**
     * Invoked after confirming that the invocation must be retried and the
     * retry context is determined.
     *
     * @param retryExecutionContext the context to use when retrying
     */
    abstract void onRetry(ExecutionContext retryExecutionContext);
}
