/* **********************************************************
 * Copyright (c) 2012-2013, 2017, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.bindings;

import java.util.HashMap;
import java.util.Map;

import com.vmware.vapi.bindings.Structure;
import com.vmware.vapi.bindings.convert.ConverterException;
import com.vmware.vapi.bindings.type.Type;
import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.data.DataValue;

/**
 * Converts values between Java data model and API runtime data model.
 */
public interface TypeConverter {

    /**
     * Context object passed during conversion of complex data structures from
     * Java binding type to {@link DataValue}.
     * <p>
     * This object provides access to the request {@link ExecutionContext} and
     * provides placeholder for additional values that may be computed in
     * individual converters.
     * <p>
     * This allows to pass the request data and also any derived settings that
     * can be cached between conversions of individual objects. For example once
     * localization preferences are computed they can be cached and reused.
     */
    public class ConversionContext {
        private final ExecutionContext ec;
        private Map<String, Object> state;

        public ConversionContext() {
            this(ExecutionContext.EMPTY);
        }

        public ConversionContext(ExecutionContext ec) {
            this.ec = ec;
        }

        public ExecutionContext getExecutionContext() {
            return ec;
        }

        public void setField(String key, Object data) {
            if (state == null) {
                state = new HashMap<>();
            }
            state.put(key, data);
        }

        public Object getField(String key) {
            if (state == null) {
                return null;
            }
            return state.get(key);

        }
    }

    /**
     * <p>
     * Converts an API runtime value (i.e. a DataValue) to a Java object.
     * </p>
     *
     * @param value     vAPI data value; must not be null
     * @param type      type information which describes the data value
     * @return Java object which represents the specified vAPI value
     * @throws ConverterException if an error occurs while converting
     */
    public <T> T convertToJava(DataValue value, Type type);

    /**
     * Converts a Java object to an API runtime value (i.e. a DataValue).
     *
     * @param obj Java object; can be {@code null} if the specified data type
     *        allows it (e.g. <code>OptionalType</code>)
     * @param type type information which describes the data value; must not be
     *        null
     * @return vAPI value which represents the specified Java object
     * @throws ConverterException if an error occurs while converting
     */
    public DataValue convertToVapi(Object obj, Type type);

    /**
     * Converts a Java object to an API runtime value (i.e. a DataValue).
     *
     * @param obj Java object; can be {@code null} if the specified data type
     *        allows it (e.g. <code>OptionalType</code>)
     * @param type type information which describes the data value; must not be
     *        null
     * @param cc provides contextual data from the request for server side
     *        conversion
     * @return vAPI value which represents the specified Java object
     * @throws ConverterException if an error occurs while converting
     */
    public DataValue convertToVapi(Object obj, Type type, ConversionContext cc);

    /**
     * Returns reference to this instance if it must to be reused for
     * constraint validation and type conversion purposes (e.g.
     * {@link Structure#_convertTo(Class)}.
     *
     * @return reference to this instance if it must be reused or
     *         {@code null} otherwise.
     */
    public TypeConverter reusableThis();
}
