/* **********************************************************************************
 * Copyright (c) 2018, 2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************************************/

package com.vmware.vapi.internal.bindings;

import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.GregorianCalendar;

import com.vmware.vapi.data.StringValue;
import com.vmware.vapi.internal.bindings.convert.PrimitiveConverter;
import com.vmware.vapi.internal.bindings.convert.impl.ConvertUtil;
import com.vmware.vapi.internal.bindings.convert.impl.GregorianCalendarConverter;
import com.vmware.vapi.internal.util.Rfc3339DateTimeConverter;

/**
 * Converts <em>DateTime</em> type to {@link java.time.ZonedDateTime}.
 * <p>
 * {@link PrimitiveConverter} implementation for DateTime. See also
 * {@link GregorianCalendarConverter}.
 */
public class ZonedDateTimeTimeConverter
        implements PrimitiveConverter<StringValue> {
    private GregorianCalendarConverter calendarConverter;

    /**
     * Creates {@link ZonedDateTimeTimeConverter} instance.
     * @param calendarConverter which is used to convert from {@link StringValue} to {@link GregorianCalendar}
     * and from {@link GregorianCalendar} to {@link StringValue}.
     */
    public ZonedDateTimeTimeConverter(GregorianCalendarConverter calendarConverter) {
        this.calendarConverter = calendarConverter;
    }

    @Override
    public ZonedDateTime fromValue(StringValue value) {
        GregorianCalendar calendar = calendarConverter.fromValue(value);
        return calendar.toZonedDateTime();
    }

    @Override
    public StringValue toValue(Object primitiveBinding) {
        ZonedDateTime zonedDateTime = ConvertUtil
                .narrowType(primitiveBinding, ZonedDateTime.class);
        ZonedDateTime timeZ = zonedDateTime.withZoneSameInstant(ZoneId.of("Z"));
        return calendarConverter.toValue(GregorianCalendar.from(timeZ));
    }

}
