/* **********************************************************
 * Copyright (c) 2013, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.bindings.convert;

import com.vmware.vapi.bindings.convert.ConverterException;
import com.vmware.vapi.bindings.type.Type;
import com.vmware.vapi.data.ConstraintValidationException;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.internal.bindings.TypeConverter;
import com.vmware.vapi.internal.bindings.TypeConverter.ConversionContext;

/**
 * Converter for a single IDL type.
 *
 * @param <V> Type of values that the converter accepts.
 * @param <T> The IDL type supported by this converter.
 */
public interface UniTypeConverter<V extends DataValue, T extends Type> {

    /**
     * Converts an API runtime data value to a binding object.
     *
     * @param value API runtime data value; must not be <code>null</code>
     * @param declaredType the type of the field as it appears in the IDL
     *                     declaration; must not be <code>null</code>
     * @param typeConverter converter which supports all IDL types and can be
     *                      used to convert elements of composite values (e.g.
     *                      fields of a structure or elements of a list); must
     *                      not be <code>null</code>
     * @return binding object; can be <code>null</code> if it makes sense for
     *         the converter implementation
     * @throws IllegalArgumentException if the contract described here is
     *                                  violated
     * @throws ConverterException general conversion error
     * @throws ConstraintValidationException if the value does not satisfy the
     *         constraints associated with the field type(e.g.
     *         <code>HasFieldsOf</code> constraint is violated)
     */
    Object fromValue(V value, T declaredType, TypeConverter typeConverter);

    /**
     * Converts a binding object to an API runtime data value.
     *
     * @param binding binding object; can be <code>null</code> if it makes
     *                sense for the converter implementation
     * @param declaredType the type of the field as it appears in the IDL
     *                     declaration; must not be <code>null</code>
     * @param typeConverter converter which supports all IDL types and can be
     *                      used to convert elements of composite values (e.g.
     *                      fields of a structure or elements of a list); must
     *                      not be <code>null</code>
     * @param cc conversion contextual data
     * @return API runtime data value; must not be <code>null</code>
     * @throws IllegalArgumentException if the contract described here is
     *                                  violated
     * @throws ConverterException general conversion error
     */
    DataValue toValue(Object binding,
                      T declaredType,
                      TypeConverter typeConverter,
                      ConversionContext cc);
}
