/* **********************************************************
 * Copyright 2013-2014 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/

/*
 * CanonicalizationUtil.java --
 *
 *      Utility methods for JSON canonicalization.
 */
package com.vmware.vapi.internal.dsig.json;

import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.Locale;

/**
 * Utility methods for JSON canonicalization.
 */
public final class CanonicalizationUtil {

    // keep DecimalFormat instance per thread, to reduce object constructions
    // and keep it thread-safe
    private static final ThreadLocal<DecimalFormat> decimalFormatPerThread =
        new ThreadLocal<DecimalFormat>() {
            @Override
            protected DecimalFormat initialValue() {
                DecimalFormatSymbols symbols = new DecimalFormatSymbols(Locale.US);
                DecimalFormat df = new DecimalFormat("0.0E0", symbols);
                df.setMaximumFractionDigits(Integer.MAX_VALUE);
                return df;
            }
    };

    /**
     * Returns JSON canonical form for <code>double</code> value
     * represented as <code>String</code>.
     *
     * <p>JSON canonical form for double is defined as: exactly one digit
     * before the decimal point followed by at least one digit after the
     * decimal point followed by "E" followed by at least one digit. E.g.
     *
     * <pre>
     *    -1.234E5
     *    1.0E-130
     *    0.0E0
     * </pre>
     *
     * @param value double value to be canonicalized
     * @return the JSON canonical representation for <code>value</code>
     * @throws IllegalArgumentException if <code>value</code> is NaN or
     *                                  Infinity
     */
    public static String canonicalizeDouble(double value) {
        if (Double.isInfinite(value) || Double.isNaN(value)) {
            throw new IllegalArgumentException("JSON does not permit numeric" +
                " values that cannot be represented as sequences of" +
                " digits (such as Infinity and NaN)");
        }
        return decimalFormatPerThread.get().format(value);
    }

}
