/* **********************************************************
 * Copyright (c) 2015-2018, 2020, 2022 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol;

import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.http.client.config.RequestConfig;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonFactory;
import com.vmware.vapi.bindings.Service;
import com.vmware.vapi.bindings.Structure;
import com.vmware.vapi.bindings.StubConfigurationBase;
import com.vmware.vapi.bindings.type.ErrorType;
import com.vmware.vapi.bindings.type.MapType;
import com.vmware.vapi.bindings.type.StructType;
import com.vmware.vapi.client.Configuration;
import com.vmware.vapi.core.ApiProvider;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.StringValue;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.internal.ClassLoaderUtil;
import com.vmware.vapi.internal.bindings.CompletionStageService;
import com.vmware.vapi.internal.bindings.OperationDef;
import com.vmware.vapi.internal.bindings.TypeConverter;
import com.vmware.vapi.internal.bindings.TypeConverterImpl;
import com.vmware.vapi.internal.bindings.ZonedDateTimeConverterFactory;
import com.vmware.vapi.internal.bindings.convert.ConverterFactory;
import com.vmware.vapi.internal.bindings.convert.NameToTypeResolver;
import com.vmware.vapi.internal.bindings.convert.PrimitiveConverter;
import com.vmware.vapi.internal.bindings.convert.UniTypeConverter;
import com.vmware.vapi.internal.bindings.convert.impl.DefaultConverterFactory;
import com.vmware.vapi.internal.bindings.convert.impl.JavaClassStructConverter;
import com.vmware.vapi.internal.bindings.convert.impl.JavaUtilCalendarRfc3339DateTimeConverter;
import com.vmware.vapi.internal.bindings.convert.impl.JavaUtilMapStructValueMapConverter;
import com.vmware.vapi.internal.bindings.convert.impl.MapBasedNameToTypeResolver;
import com.vmware.vapi.internal.client.Protocol;
import com.vmware.vapi.internal.protocol.client.rest.BindingsOperationRestMetadataProvider;
import com.vmware.vapi.internal.protocol.client.rest.BodyConverter;
import com.vmware.vapi.internal.protocol.client.rest.DefaultBodyConverter;
import com.vmware.vapi.internal.protocol.client.rest.DefaultRequestExecutorFactory;
import com.vmware.vapi.internal.protocol.client.rest.DefaultResponseParser;
import com.vmware.vapi.internal.protocol.client.rest.ErrorConverter;
import com.vmware.vapi.internal.protocol.client.rest.OperationRestMetadataProvider;
import com.vmware.vapi.internal.protocol.client.rest.RequestBuilderFactory;
import com.vmware.vapi.internal.protocol.client.rest.RequestExecutor;
import com.vmware.vapi.internal.protocol.client.rest.RequestExecutorFactory;
import com.vmware.vapi.internal.protocol.client.rest.ResponseParser;
import com.vmware.vapi.internal.protocol.client.rest.RestClientApiProvider;
import com.vmware.vapi.internal.protocol.client.rest.RestProtocolConnection;
import com.vmware.vapi.internal.protocol.client.rest.RestRequestBuilderFactory;
import com.vmware.vapi.internal.protocol.client.rest.StatusCodeErrorConverter;
import com.vmware.vapi.internal.protocol.client.rpc.RestTransport;
import com.vmware.vapi.internal.protocol.client.rpc.http.ApacheAsyncClientRestTransport;
import com.vmware.vapi.internal.protocol.client.rpc.http.ApacheClientRestTransport;
import com.vmware.vapi.internal.protocol.client.rpc.http.ConnectionMonitor.CleanableConnectionPool;
import com.vmware.vapi.internal.protocol.common.DirectDeserializer;
import com.vmware.vapi.internal.protocol.common.DirectSerializer;
import com.vmware.vapi.internal.protocol.common.http.UrlUtil;
import com.vmware.vapi.internal.protocol.common.json.JsonDirectDeserializer;
import com.vmware.vapi.internal.protocol.common.json.JsonDirectSerializer;
import com.vmware.vapi.protocol.ProtocolConnection;
import com.vmware.vapi.protocol.client.http.RequestPreProcessor;

public class RestProtocol implements Protocol {
    private static final String DEFINITIONS_CLASS_NAME_SUFFIX = "Definitions";
    private static final String OPERATION_DEFS_FIELD_NAME = "__operationDefs";
    private static final Logger LOGGER =
            LoggerFactory.getLogger(RestProtocol.class);

    private static final Protocol INSTANCE = new RestProtocol();

    private RestProtocol() {
    }

    public final static String REST_REQUEST_AUTHENTICATOR_CFG = "rest.request.authenticator";
    public final static String REST_REQUEST_EXECUTOR_CFG = "rest.request.executor";

    public static Protocol getInstance() {
        return INSTANCE;
    }

    @Override
    public ApiProviderCardinality getApiProviderCardinality() {
        return ApiProviderCardinality.PER_STUB;
    }

    @Override
    public ProtocolConnection getProtocolConnection(String url,
                                                    Configuration config) {
        return new RestProtocolConnection(buildRestClientApiProvider(url, config));
    }

    @Override
    public <T extends Service> TypeConverter getTypeConverter(StubConfigurationBase stubConfig, Class<T> vapiIface) {
        Set<ErrorType> stubConfigErrors = null;
        if (stubConfig != null) {
            stubConfigErrors = stubConfig.getErrorTypes();
        }
        Map<String, StructType> errorTypes =
                MapBasedNameToTypeResolver.augmentWithStandardErrors(stubConfigErrors);
        NameToTypeResolver errorTypeResolver =
                new MapBasedNameToTypeResolver(errorTypes);
        ConverterFactory converterFactory = this.getConverterFactory(vapiIface, errorTypeResolver);

        return new TypeConverterImpl(converterFactory,
                                     true,   // "permissive" mode (deprecated, TODO: remove)
                                     true);  // reuse instance for validation
    }

    <T extends Service> ConverterFactory getConverterFactory(Class<T> vapiIface,
                                                             NameToTypeResolver nameToTypeResolver) {
        if (CompletionStageService.class.isAssignableFrom(vapiIface)) {
            return new ZonedDateTimeConverterFactory(new RestStubConverterFactory(nameToTypeResolver));
        }

        return new RestStubConverterFactory(nameToTypeResolver);
    }

    RestTransport resolveTransport(Configuration config) {
        RequestConfig defaultRequestConfig = resolveDefaultRequestConfiguration(config);
        CleanableConnectionPool connectionPool = resolveConnectionPoolConfiguration(config);

        CloseableHttpClient apacheHttpClient = config
                .getProperty(BIO_HTTP_CLIENT_MONICKER,
                             CloseableHttpClient.class);
        if (apacheHttpClient != null) {
            return new ApacheClientRestTransport(apacheHttpClient,
                                                 defaultRequestConfig,
                                                 connectionPool);
        }

        CloseableHttpAsyncClient apacheHttpAsyncClient = config
                .getProperty(NIO_HTTP_CLIENT_MONICKER,
                             CloseableHttpAsyncClient.class);
        if (apacheHttpAsyncClient != null) {
            return new ApacheAsyncClientRestTransport(apacheHttpAsyncClient,
                                                      defaultRequestConfig,
                                                      connectionPool);
        }

        // TODO: contract; allow creation of new apache client here
        throw new RuntimeException("Didn't receive apache client");
    }

    RequestConfig resolveDefaultRequestConfiguration(Configuration config) {
        return config.getProperty(DEFAULT_REQUEST_CONFIG_MONICKER,
                                  RequestConfig.class);
    }

    CleanableConnectionPool resolveConnectionPoolConfiguration(Configuration config) {
        return config.getProperty(VAPI_CONNECTION_POOL,
                                  CleanableConnectionPool.class);
    }

    ApiProvider buildRestClientApiProvider(String url, Configuration config) {
        RestTransport restTransport = resolveTransport(config);
        JsonFactory jsonFactory = new JsonFactory();
        ResponseParser responseParser = buildResponseParser(jsonFactory);

        RestClientApiProvider provider =
                new RestClientApiProvider(buildRequestExecutor(restTransport, responseParser, config),
                                          buildRequestBuilderFactory(url, jsonFactory, config));

        RequestPreProcessor authenticator = buildRestAuthenticator(config);
        if (authenticator != null) {
            provider.setPreProcessors(Arrays.asList(authenticator));
        }

        return provider;
    }

    private RequestExecutor buildRequestExecutor(RestTransport transport,
                                                 ResponseParser parser,
                                                 Configuration config) {
        RequestExecutorFactory executorFactory =
                config.getProperty(REST_REQUEST_EXECUTOR_CFG,
                                   RequestExecutorFactory.class);

        if (executorFactory == null) {
            executorFactory = new DefaultRequestExecutorFactory();
        }

        return executorFactory.createRequestExecutor(transport, parser);
    }

    protected RequestPreProcessor buildRestAuthenticator(Configuration config) {
        return config.getProperty(REST_REQUEST_AUTHENTICATOR_CFG,
                                  RequestPreProcessor.class);
    }

    RequestBuilderFactory buildRequestBuilderFactory(String url,
                                                     JsonFactory jsonFactory,
                                                     Configuration config) {
        return new RestRequestBuilderFactory(url,
                                           buildSerializer(jsonFactory),
                                           buildRestMetadataProvider(config),
                                           new UrlUtil());
    }

    ResponseParser buildResponseParser(JsonFactory jsonFactory) {
        DirectDeserializer jsonDeserializer = buildDeserializer(jsonFactory);
        BodyConverter bodyConverter = new DefaultBodyConverter(jsonDeserializer);
        ErrorConverter errorConverter = new StatusCodeErrorConverter(bodyConverter);

        return new DefaultResponseParser(bodyConverter, errorConverter);

    }

    DirectSerializer buildSerializer(JsonFactory jsonFactory) {
        return new JsonDirectSerializer(jsonFactory);
    }

    OperationRestMetadataProvider buildRestMetadataProvider(Configuration config) {
        // TODO: rest native: must allow other OperationRestMetadataProviders
        //       to be configured for the dynamic client (no bindings) case
        return new BindingsOperationRestMetadataProvider(resolveRestMetadata(config));
    }

    DirectDeserializer buildDeserializer(JsonFactory jsonFactory) {
        return new JsonDirectDeserializer(jsonFactory);
    }

    List<OperationDef> resolveRestMetadata(Configuration config) {
        Class<?> vapiIface = config.getProperty(VAPI_INTERFACE_MONICKER, Class.class);

        if (vapiIface == null) {
            String errorMsg = "Failed to load REST metadata for the service, "
                    + "since its Class instance is not specified.";
            LOGGER.error(errorMsg);
            throw new RuntimeException(errorMsg);
        }

        String definitionsClassName = vapiIface.getName() + DEFINITIONS_CLASS_NAME_SUFFIX;
        try {
            ClassLoader cl = ClassLoaderUtil.getServiceClassLoader();
            Class<?> definitionsClass = cl.loadClass(definitionsClassName);

            Field operDefsField = definitionsClass.getDeclaredField(OPERATION_DEFS_FIELD_NAME);
            operDefsField.setAccessible(false);
            Object value = operDefsField.get(null);

            @SuppressWarnings("unchecked")
            List<OperationDef> operDefValue = (List<OperationDef>) value;
            return operDefValue;
        } catch (Exception ex) {
            String errorMsg = "Failed to load REST metadata for target service";
            LOGGER.error(errorMsg);
            throw new RuntimeException(errorMsg, ex);
        }
    }

    /**
     * {@code ConverterFactory} for REST protocol.
     *
     * <p>The REST specific converters this factory instantiates
     * <ul>
     * <li>skip fields with unset {@code OptionalValue} values
     * <li>handle Maps represented as {@code StructValue}
     * <li>handle <b>RFC 3339</b> dateTime values
     * </ul>
     */
    static final class RestStubConverterFactory extends DefaultConverterFactory {
        // true - skip unset optional fields converting to DataValue
        private final UniTypeConverter<StructValue, StructType> skipUnsetFieldsStructConverter =
                new JavaClassStructConverter<>(StructValue.class, Structure.class, true);

        private final UniTypeConverter<DataValue, MapType> structMapConverter =
                new JavaUtilMapStructValueMapConverter();

        private final JavaUtilCalendarRfc3339DateTimeConverter rfc3339DateTimeConverter =
                new JavaUtilCalendarRfc3339DateTimeConverter();

        RestStubConverterFactory(NameToTypeResolver nameToTypeResolver) {
            super(nameToTypeResolver);
        }

        @Override
        public UniTypeConverter<StructValue, StructType> getStructConverter() {
            return skipUnsetFieldsStructConverter;
        }

        @Override
        public UniTypeConverter<DataValue, MapType> getMapConverter() {
            return structMapConverter;
        }

        @Override
        public PrimitiveConverter<StringValue> getDateTimeConverter() {
            return rfc3339DateTimeConverter;
        }
    }
}
