/* **********************************************************
 * Copyright (c) 2017 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.client.rest.authn;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.vmware.vapi.protocol.HttpConfiguration;

/**
 * Utility class for fetching authentication token from HTTP URL
 * of CSP (VMware Common SaaS Platform).
 *
 * <p><b>WARNING: </b> This class is subject to change.
 */
public class CspTokenProvider extends HttpTokenProviderBase {
    private static final Logger logger =
            LoggerFactory.getLogger(CspTokenProvider.class);

    /**
     * Constructor.
     *
     * @param url target URL to be used for obtaining OAuth token
     * @param httpConfig HTTP configuration for executing request to the URL; may be {@code null}
     */
    public CspTokenProvider(String url, HttpConfiguration httpConfig) {
        super(url, httpConfig);
    }

    @Override
    protected HttpUriRequest buildTokenRequest(String username, String password)
            throws JsonProcessingException {
        logger.debug("Getting CSP authentication token using username and password.");
        Map<String, String> requestBody = new HashMap<>();
        requestBody.put("username", username);
        requestBody.put("password", password);
        return buildRequest(requestBody);
    }

    @Override
    protected HttpUriRequest buildTokenRequest(String accessKey)
            throws JsonProcessingException {
        logger.debug("Getting CSP authentication token using access-key.");
        Map<String, String> requestBody = new HashMap<>();
        requestBody.put("key", accessKey);
        return buildRequest(requestBody);
    }

    @Override
    protected String extractToken(HttpResponse response) throws IOException {
        if (response.getStatusLine().getStatusCode() != 200) {
            throw new FetchTokenException("Failed to obtain token. Received HTTP status code: "
                                           + response.getStatusLine().getStatusCode());
        }

        try (InputStream inStream = response.getEntity().getContent()) {
            ObjectMapper jsonMapper = new ObjectMapper();
            Map<String, String> result =
                    jsonMapper.readValue(inStream, new TypeReference<Map<String, String>>(){ });
            if (result.containsKey("cspAuthToken")) {
                return result.get("cspAuthToken");
            }
            throw new FetchTokenException("Failed to obtain token. No 'cspAuthToken' "
                    + "property found in response.");
        }
    }

    private HttpPost buildRequest(Map<String, String> requestBody) throws JsonProcessingException {
        HttpPost request = new HttpPost(getUrl());
        ObjectMapper jsonMapper = new ObjectMapper();
        String requestBodyString = jsonMapper.writeValueAsString(requestBody);
        request.setEntity(new StringEntity(requestBodyString, ContentType.APPLICATION_JSON));
        return request;
    }
}
