/* **********************************************************
 * Copyright (c) 2017, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.client.rest.authn;

import java.io.Closeable;
import java.io.IOException;

import org.apache.http.HttpResponse;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.impl.client.CloseableHttpClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.internal.protocol.client.rpc.http.ApacheBioHttpClientBuilder;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.protocol.HttpConfiguration;

/**
 * Abstract base class for authentication token providers which use HTTP
 * to obtain the token and user/password credentials to authenticate to
 * the token provider endpoint.
 */
public abstract class HttpTokenProviderBase implements Closeable {
    private static final Logger logger =
            LoggerFactory.getLogger(HttpTokenProviderBase.class);

    private final String url;
    private final CloseableHttpClient client;

    public HttpTokenProviderBase(String url, HttpConfiguration httpConfig) {
        Validate.notNull(url);

        this.url = url;
        if (httpConfig == null) {
            httpConfig = new HttpConfiguration.Builder().getConfig();
        }
        this.client = new ApacheBioHttpClientBuilder().buildAndConfigure(httpConfig);
    }

    /**
     * Releases resource. This instance can not be used to request OAuth tokens after
     * this method is invoked.
     */
    @Override
    public void close() throws IOException {
        if (client != null) {
            client.close();
        }
    }

    /**
     * Obtain authentication (access) token from the authorization endpoint.
     *
     * @param username user name
     * @param password password
     * @return the fetched authentication token
     * @throws FetchTokenException if unable to fetch token successfully
     */
    public String fetchToken(String username, String password) {
        Validate.notNull(username);
        Validate.notNull(password);
        try {
            HttpUriRequest request = buildTokenRequest(username, password);
            try (CloseableHttpResponse response = client.execute(request)) {
                String accessToken = extractToken(response);
                if (accessToken == null) {
                    throw new FetchTokenException("Received a 'null' token");
                }
                return accessToken;
            }
        } catch (Exception e) {
            String msg =
                    String.format("Got %s while retrieving the access token.",
                                  e.getClass().getName());
            logger.error(msg, e);
            throw new FetchTokenException(msg, e);
        }
    }

    /**
     * Obtain authentication (access) token from the authorization endpoint.
     *
     * @param accessKey user's access key
     * @return the fetched authentication token
     * @throws FetchTokenException if unable to fetch token successfully
     */
    public String fetchToken(String accessKey) {
        Validate.notNull(accessKey);
        try {
            HttpUriRequest request = buildTokenRequest(accessKey);
            try (CloseableHttpResponse response = client.execute(request)) {
                String accessToken = extractToken(response);
                if (accessToken == null) {
                    throw new FetchTokenException("Received a 'null' token");
                }
                return accessToken;
            }
        } catch (Exception e) {
            String msg =
                    String.format("Got %s while executing session create request.",
                                  e.getClass().getName());
            logger.error(msg, e);
            throw new FetchTokenException(msg, e);
        }
    }

    protected String getUrl() {
        return this.url;
    }

    protected abstract HttpUriRequest buildTokenRequest(String username,
                                                        String password) throws Exception;

    protected abstract HttpUriRequest buildTokenRequest(String accessKey) throws Exception;

    protected abstract String extractToken(HttpResponse response) throws Exception;
}
