/* **********************************************************
 * Copyright (c) 2015-2016, 2018 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.client.rpc;

import java.util.List;
import java.util.Map;

/**
 * Defines a simplified interface to HTTP request abstraction for use by the REST client code. This
 * can be implemented over different HTTP libraries.
 */
public interface HttpRequest {

    public enum HttpMethod {
        // All RFC 7231 methods but CONNECT and TRACE
        // RFC 5789 - PATCH
        GET, HEAD, POST, PUT, PATCH, DELETE, OPTIONS;
    }

    /**
     * Handler used to complete an HTTP request execution and report the HTTP response or error.
     */
    public interface HttpResponseHandler {

        /**
         * On a successful HTTP request execution report the response.
         *
         * @param response of the HTTP request execution.
         */
        void onResult(HttpResponse response);

        /**
         * On an unsuccessful HTTP request execution report the error.
         *
         * @param ex error of the HTTP request execution.
         */
        void onError(Exception ex);
    }

    public String getUrl();

    /**
     * @param url of the HTTP request. Must not be <code>null</code>.
     */
    public void setUrl(String url);

    public HttpMethod getMethod();

    /**
     * @param method of the HTTP request. Must not be <code>null</code>.
     */
    public void setMethod(HttpMethod method);

    // TODO: use InputStream instead of byte[];
    public byte[] getBody();

    /**
     * @param body of the HTTP request.
     */
    public void setBody(byte[] body);

    public Map<String, List<String>> getHeaders();

    public void setHeaders(Map<String, List<String>> headers);

    /**
     * Add a header to the HTTP request.
     *
     * @param name of header. Must not be <code>null</code>.
     * @param value of header. Must no tbe <code>null</code>.
     */
    public void addHeader(String name,
                          String value);

    /**
    * Determines the amount of time in milliseconds the client will wait for a
    * response from the server before timing out.
    *
    * <p>For the HTTP 1.1 transport protocol this value maps to the
    * {@code SO_TIMEOUT}.</p>
    *
    * @return the number of milliseconds the client will wait for a response
    *         from the server; a value of zero is interpreted as an infinite
    *         timeout; a value of {@code null} is interpreted as not specified
    *         at this level (configuration default)
    */
    public Integer getReadTimeout();
}
