/* **********************************************************
 * Copyright (c) 2023 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.client.rpc.http.handle;

import java.nio.BufferOverflowException;
import java.nio.ByteBuffer;

import org.apache.http.nio.util.ByteBufferAllocator;
import org.apache.http.nio.util.SimpleInputBuffer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * An expandable byte buffer, extension of the apache {@link SimpleInputBuffer},
 * which can impose a limit on the maximum size of the input it can consume
 */
public class CappedInputBuffer extends SimpleInputBuffer {

    private static final Logger logger = LoggerFactory
            .getLogger(CappedInputBuffer.class);

    private final int maxBuffSize;
    private final ByteBufferAllocator allocator;

    /**
     * @param maxBuffSize the maximum size of the buffer that is allowed.<br>
     *        Note that during memory allocation a memory 1.5 the size of the
     *        needed one is temporarily allocated
     * @param initBufferSize the initial buffer size
     * @param allocator to be used to allocate memory
     */
    CappedInputBuffer(final int maxBufferSize,
                             final int initBufferSize,
                             final ByteBufferAllocator allocator) {
        super(initBufferSize, allocator);
        if (initBufferSize > maxBufferSize) {
            throw new IllegalArgumentException("initBufferSize "
                                               + initBufferSize
                                               + " cannot be bigger than maxBufferSize "
                                               + maxBufferSize);
        }
        if (maxBufferSize <= 0) {
            throw new IllegalArgumentException("maxBufferSIze must be a positive number");
        }
        this.maxBuffSize = maxBufferSize;
        this.allocator = allocator;
    }

    @Override
    protected void expand() throws BufferOverflowException {
        int newCapacity = this.buffer.capacity() << 1;
        if (newCapacity < 0) {
            //copied from the parent class
            final int vmBytes = Long.SIZE >> 3;
            final int javaBytes = 8; // this is to be checked when the JVM version changes
            @SuppressWarnings("unused") // we really need the 8 if we're going to make this foolproof
            final int headRoom = (vmBytes >= javaBytes) ? vmBytes : javaBytes;

            newCapacity = Integer.MAX_VALUE - headRoom;
            if (newCapacity <= this.buffer.capacity()) {
                throw new BufferOverflowException();
            }
        }
        expandCapacity(newCapacity);
    }

    private void expandCapacity(final int capacity) {
        if (capacity > maxBuffSize) {
            logger.warn("Buffer size exceeded. Allowed: " +maxBuffSize + " requested: "+capacity);
            throw new BufferOverflowException();
        }
        final ByteBuffer oldbuffer = this.buffer;
        this.buffer = allocator.allocate(capacity);
        oldbuffer.flip();
        this.buffer.put(oldbuffer);
    }
}