/* **********************************************************
 * Copyright (c) 2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.protocol.client.rpc.http.handle;

import java.io.IOException;

import org.apache.http.HttpException;
import org.apache.http.HttpResponse;
import org.apache.http.nio.ContentDecoder;
import org.apache.http.nio.IOControl;
import org.apache.http.nio.protocol.HttpAsyncResponseConsumer;
import org.apache.http.protocol.HttpContext;

/**
 * Decorator which wraps an existing response consumer. Utility base class.
 * The decorated consumer is supposed to be configured exactly once - either
 * in the constructor or in {@link #responseReceived(HttpResponse)} when the
 * Content-Type header is received.
 */
public abstract class NioDecoratorConsumer implements
        HttpAsyncResponseConsumer<HttpResponse> {

    protected volatile HttpAsyncResponseConsumer<HttpResponse> decorated;

    @Override
    public void responseReceived(HttpResponse response) throws IOException,
            HttpException {
        if (decorated != null) {
            decorated.responseReceived(response);
        }
    }

    @Override
    public void consumeContent(ContentDecoder decoder, IOControl ioctrl)
            throws IOException {
        if (decorated != null) {
            decorated.consumeContent(decoder, ioctrl);
        }
    }

    @Override
    public void responseCompleted(HttpContext context) {
        if (decorated != null) {
            decorated.responseCompleted(context);
        }
    }

    @Override
    public void failed(Exception ex) {
        if (decorated != null) {
            decorated.failed(ex);
        }
    }

    @Override
    public Exception getException() {
        if (decorated != null) {
            return decorated.getException();
        } else {
            return null;
        }
    }

    @Override
    public HttpResponse getResult() {
        if (decorated != null) {
            return decorated.getResult();
        } else {
            return null;
        }
    }

    @Override
    public boolean isDone() {
        if (decorated != null) {
            return decorated.isDone();
        } else {
            return false;
        }
    }

    @Override
    public void close() throws IOException {
        if (decorated != null) {
            decorated.close();
        }
    }

    @Override
    public boolean cancel() {
        if (decorated != null) {
            return decorated.cancel();
        } else {
            return true;
        }
    }
}